﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesAccessPoint.h>


namespace nn
{
namespace nifm
{
    //! @name 接続情報取得 API
    //! @{

    /**
     * @brief 接続に利用されている接続設定を取得します。
     *
     * @param[out] pOutNetworkProfileData 取得される接続設定を格納するバッファへのポインタ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutNetworkProfileData != nullptr
     *
     * @details
     *  現在接続中でない場合は失敗します。失敗時に得られる値は不定です。
     */
    nn::Result GetCurrentNetworkProfile( NetworkProfileData* pOutNetworkProfileData ) NN_NOEXCEPT;

    /**
     * @brief 接続に利用されているアクセスポイント情報を取得します。
     *
     * @param[out] pOutAccessPointData 取得されるアクセスポイント情報を格納するバッファへのポインタ
     *
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     *   - ライブラリが初期化済み
     *   - pOutAccessPointData != nullptr
     *
     * @details
     *  現在の接続が、無線かつインターネット通信でない場合は失敗します。失敗時に得られる値は不定です。
     *  アクセスポイントが複数の認証・暗号化方式をサポートする場合、この API で取得できるのは
     *  nifm::GetScanData() で得られる値と同様です。
     *  現在使用されている認証・暗号化方式が必要な場合 nifm::GetCurrentNetworkProfile() を使用してください。
     */
    nn::Result GetCurrentAccessPoint( AccessPointData* pOutAccessPointData ) NN_NOEXCEPT;

    //! @}

    //! @name ネットワーク接続設定関連 API
    //! @{

    /**
     * @brief 利用可能なネットワークインタフェースの一覧を取得します。
     *
     * @param[out] pOutNetworkInterfaceInfo 取得されるネットワークインタフェース情報を格納するバッファへのポインタ
     * @param[out] pOutCount                実際に取得されたインタフェース数
     * @param[in]  bufferCount              インタフェース数で表される、用意されたバッファのサイズ
     * @param[in]  filterFlags              @ref nn::nifm::EnumerateNetworkInterfacesFilter の論理和
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutNetworkInterfaceInfo != nullptr
     *   - pOutCount != nullptr
     *
     * @details
     *  出力用バッファ pOutNetworkInterfaceInfo には、bufferCount 以下の個数しか格納されません。
     *  すべての出力がバッファに格納できないような場合、pOutCount には bufferCount よりも大きい値が入ります。
     */
    nn::Result EnumerateNetworkInterfaces( NetworkInterfaceInfo* pOutNetworkInterfaceInfo, int* pOutCount, int bufferCount, nn::Bit32 filterFlags ) NN_NOEXCEPT;

    /**
    * @brief 接続設定の種類を指定してネットワーク接続設定の基本情報一覧を取得します。
    *
    * @param[out] pOutNetworkProfileBasicInfo 取得されるネットワーク接続設定の基本情報を格納するバッファへのポインタ
    * @param[out] pOutCount                   実際に取得された基本情報数
    * @param[in]  bufferCount                 基本情報数で表される、用意されたバッファのサイズ
    * @param[in]  typeFlags                   @ref nn::nifm::NetworkProfileType の論理和
    *
    * @retresult
    *   @handleresult{
    *     nn::ResultSuccess, 処理に成功しました。
    *   }
    * @endretresult
    *
    * @pre
    *   - ライブラリが初期化済み
    *   - pOutNetworkProfileBasicInfo != nullptr
    *   - pOutCount != nullptr
    *
    * @details
    *  出力用バッファ pOutNetworkProfileBasicInfo には、bufferCount 以下の個数しか格納されません。
    *  すべての出力がバッファに格納できないような場合、pOutCount には bufferCount よりも大きい値が入ります。@n
    *
    *  取得される基本情報 nn::nifm::NetworkProfileBasicInfo には、ネットワーク接続設定の ID が含まれます。
    *  より詳細な情報が必要な場合は、ネットワーク接続設定 ID を nn::nifm::GetNetworkProfile 関数に渡して取得してください。
    *
    * @platformbegin{Windows}
    *  Windows 版において、取得されるネットワーク接続設定は、Windows 自身のネットワーク接続設定とは別物です。
    * @platformend
    */
    nn::Result EnumerateNetworkProfiles(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, int* pOutCount, int bufferCount, nn::Bit8 typeFlags ) NN_NOEXCEPT;

    /**
     * @brief ID で指定されたネットワーク接続設定の詳細情報を取得します。
     *
     * @param[out] pOutNetworkProfileData 取得されるネットワーク接続設定を格納するバッファへのポインタ
     * @param[in]  id                     取得されるネットワーク接続設定の ID
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutNetworkProfileData != nullptr
     *
     * @details
     *  指定された ID のネットワーク接続設定が存在しない場合は失敗します。失敗時に得られる値は不定です。@n
     *
     * @platformbegin{Windows}
     *  Windows 版において、取得されるネットワーク接続設定は、Windows 自身のネットワーク接続設定とは別物です。
     * @platformend
     */
    nn::Result GetNetworkProfile( NetworkProfileData* pOutNetworkProfileData, const nn::util::Uuid& id ) NN_NOEXCEPT;

    /**
     * @brief ネットワーク接続設定を追加または更新します。
     *
     * @param[out] pOutId             追加または更新された接続設定の ID です。
     * @param[in]  networkProfileData 追加または更新する接続設定情報です。@n
     *                                追加する場合は、NetworkProfileData の id に nn::nifm::InvalidUuid を指定してください。@n
     *                                更新する場合は、NetworkProfileData の id に更新対象の ID を指定してください。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutId != nullptr
     *
     * @details
     *  networkProfileData に指定された ID が、nn::nifm::InvalidUuid ではなく存在する接続設定の ID でもない場合、失敗が返ります。
     *  存在する接続設定の ID が指定された場合、その設定が更新されると同時に ID も再生成され pOutId として返されます。@n
     *  また、本関数でテンポラリ接続設定を更新することはできません。
     *
     * @platformbegin{Windows}
     *  Windows 版において、追加または更新されるネットワーク接続設定は、Windows 自身のネットワーク接続設定とは別物であり、それらが影響を受けることはありません。
     * @platformend
     */
    nn::Result SetNetworkProfile( nn::util::Uuid* pOutId, const NetworkProfileData& networkProfileData ) NN_NOEXCEPT;

    /**
     * @brief ID で指定されたネットワーク接続設定を削除します。
     *
     * @param[in] id 削除するネットワーク接続設定の ID
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  nn::nifm::SetNetworkProfile によって追加された接続設定を削除することができます。
     *  networkProfileData に指定された ID の接続設定が存在しない場合、失敗が返ります。
     *
     * @platformbegin{Windows}
     *  Windows 版において、削除されるネットワーク接続設定は、Windows 自身のネットワーク接続設定とは別物であり、それらが影響を受けることはありません。
     * @platformend
     */
    nn::Result RemoveNetworkProfile( const nn::util::Uuid& id ) NN_NOEXCEPT;

    /**
     * @brief SSID リストのバージョン文字列を取得します。
     *
     * @param[out] pOutSsidListVersion  取得されるSSID リストのバージョン文字列を格納するバッファへのポインタ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutSsidListVersion != nullptr
     *
     * @details
     *
     */
    nn::Result GetSsidListVersion(SsidListVersion *pOutSsidListVersion) NN_NOEXCEPT;


    /**
     * @brief 指定されたテンポラリ接続設定と同様の設定を通常のネットワーク接続設定として保存します。
     *
     * @param[in] handle テンポラリ接続設定のハンドル
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *
     * @details
     *  この関数により、指定されたテンポラリ接続設定と同一 ID の接続設定が保存され、テンポラリ接続設定の ID は変更されます。
     *  つまり、本関数呼び出しの前後でテンポラリ接続設定の ID が変化するので、呼び出し後にテンポラリ接続設定を利用する場合には、
     *  ID を再取得する必要があります。
     */
    nn::Result PersistTemporaryNetworkProfile(NetworkProfileHandle handle) NN_NOEXCEPT;

    //! @}
}
}
