﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesRequestPrivate.h>

#include <nn/applet/applet.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_Types.h>

namespace nn
{
namespace nifm
{
namespace detail
{

class IRequest;

/**
* @brief 利用要求を表します。
*/
class RequestClient
{
    NN_DISALLOW_COPY(RequestClient);
    NN_DISALLOW_MOVE(RequestClient);

private:
    nn::sf::SharedPointer<detail::IRequest> m_pIRequest;

    nn::os::SystemEvent m_SystemEventInternal;
    nn::os::SystemEvent m_SystemEventExposed;
    nn::os::SdkRecursiveMutex m_Mutex;

    RequestState m_RequestStateCache;
    Result m_ResultCache;

private:
    void UpdateInternalState() NN_NOEXCEPT;

public:
    RequestClient(RequirementPreset requirementPreset, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;
    ~RequestClient() NN_NOEXCEPT;

    /**
     * @brief 要求を提出します。
     */
    void Submit() NN_NOEXCEPT;
    void SubmitAndWait() NN_NOEXCEPT;

    /**
     * @brief 提出した要求を取り消します。
     */
    void Cancel() NN_NOEXCEPT;

    /**
     * @brief 利用要求の状態を取得します。
     */
    RequestState GetRequestState() NN_NOEXCEPT;

    /**
     * @brief 利用要求の結果を取得します。
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /*
     * @brief 利用要求の状態変化を待ち受けるためのシステムイベントを取得します。
     */
    nn::os::SystemEvent& GetSystemEvent() NN_NOEXCEPT;

    /**
    * @brief 利用要求紐づく、提出結果に付随する追加情報を取得します。
    */
    nn::Result GetAdditionalInfo(AdditionalInfo* pOutAdditionalInfo, uint32_t* pOutRevision) NN_NOEXCEPT;

    // 以下の Set 系 API は未提出状態でなければ失敗します

    /**
    * @brief 利用要求の内容を設定します。
    */
    nn::Result SetRequirement(const Requirement& requirement) NN_NOEXCEPT;

    /**
    * @brief 利用要求の内容を取得します。
    */
    nn::Result GetRequirement(Requirement* pOutRequirement) const NN_NOEXCEPT;

    /**
    * @brief 利用要求の提出 ID を取得します。
    */
    nn::Result GetSubmitId(uint32_t *pOutSubmitId) const NN_NOEXCEPT;

    /**
    * @brief 利用要求の内容をプリセットで設定します。
    */
    nn::Result SetRequirementPreset(RequirementPreset requirementPreset) NN_NOEXCEPT;

    /**
    * @brief 指定された利用要求から内容をコピーして設定します。
    */
    nn::Result SetRequirementBySubmitId(uint32_t revision) NN_NOEXCEPT;

    /**
     * @brief 利用要求の優先度を設定します。
     */
    nn::Result SetPriority(uint8_t priority) NN_NOEXCEPT;

    /**
     * @brief 利用要求の優先度（直値）を設定します。
     */
    nn::Result SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT;

    /**
     * @brief 利用要求で利用するプロファイルを設定します。
     */
    nn::Result SetNetworkProfileId(nn::util::Uuid uuid) NN_NOEXCEPT;

    /**
     * @brief 利用要求に対する取り下げ依頼を受け取るか否かを設定します。
     */
    nn::Result SetRejectable(bool isRejectable) NN_NOEXCEPT;

    /**
     * @brief インターネット疎通確認に関する要求を設定します。
     */
    nn::Result SetConnectionConfirmationOption(ConnectionConfirmationOption connectionConfirmationOption) NN_NOEXCEPT;

    /**
     * @brief 利用要求の継続設定を指定します。
     */
    nn::Result SetPersistent(bool isPersistent) NN_NOEXCEPT;

    /**
     * @brief 利用要求の即時評価設定を指定します。
     */
    nn::Result SetInstant(bool isInstant) NN_NOEXCEPT;

    /**
     * @brief 利用要求の接続維持設定を指定します。
     */
    nn::Result SetSustainable(bool isSustainable) NN_NOEXCEPT;

    /**
     * @brief 利用要求が、排除可能な他の要求を排除してより大きい帯域を使用するか否か、を設定します。
     */
    nn::Result SetGreedy(bool isGreedy) NN_NOEXCEPT;

    /**
     * @brief 利用要求が、大きい帯域を要求する利用要求と帯域を共有可能であるか否か、を設定します。
     */
    nn::Result SetSharable(bool isSharable) NN_NOEXCEPT;

    /**
     * @brief 利用要求がスリープ遷移時にも取り下げずに維持するよう求めるか、を設定します。
     *
     * @details
     *  この設定を有効にしていても、 nn::nifm::SetRequestSocketDescriptor() で維持するべきセッションの登録をおこなっていなかった場合、スリープ遷移時にネットワーク接続が切断され、利用要求が取り下げられます。@n
     *  また、システム上の制約により接続が維持できない状況下では、この関数での設定にかかわらず、スリープ遷移時にネットワーク接続が切断され、利用要求が取り下げられることがあります。@n
     */
    nn::Result SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を登録します。
     *
     * @details
     *  スリープ中にネットワーク接続を維持するには、別途 nn::nifm::SetRequestRequirementPreset() もしくは
     *  nn::nifm::SetRequestKeptInSleep() で事前に設定をおこなう必要があります。@n
     *  @n
     *  この関数は利用要求が受理された状態で呼ばないと失敗を返します。
     */
    nn::Result RegisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を破棄します。
     *
     * @details
     *  引数に -1 を指定した場合、すべての登録を破棄します。
     */
    nn::Result UnregisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT;

    /**
     * @brief 直近の利用要求提出の結果をもとに、必要があればライブラリアプレットを作成し、必要な情報を InChannel に書き込みます。
     */
    nn::Result PrepareLibraryApplet(nn::applet::LibraryAppletHandle* pOutLibraryAppletHandle) NN_NOEXCEPT;

    /**
     * @brief 直近の利用要求提出の結果をもとにライブラリアプレットを作成するための引数を取得します。
     *
     * @details  この関数はテスト用です。
     */
    nn::Result GetLibaryAppletArgument(nn::applet::AppletId* pOutAppletId, nn::applet::LibraryAppletMode* pOutLibrayAppletMode, size_t* pOutSize, void* pOutBuffer, size_t size) NN_NOEXCEPT;
};

}
}
}

