﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>
#include <nn/nifm/nifm_TypesRequirement.h>
#include <nn/nifm/nifm_TypesClientManagement.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_IntrusiveList.h>

#include <atomic>


namespace nn
{
namespace nifm
{
namespace detail
{

// 内部用プリセット
static const RequirementPreset RequirementPreset_SustainConnection = RequirementPreset_Reserved1;

class RequestManager;
class ISignalObject;

class UserRequest : public nn::util::IntrusiveListBaseNode<UserRequest>
{
    NN_DISALLOW_COPY(UserRequest);
    NN_DISALLOW_MOVE(UserRequest);

private:
    RequestManager* m_pRequestManager;
    Requirement m_Requirement;

    ISignalObject* m_pSignalObject;
    ClientId m_ClientId;
    const nn:: Bit64 m_ClientProcessId;

    mutable nn::os::SdkMutex m_Mutex;

    RequestState m_RequestState;
    nn::Result m_Result;

    uint32_t m_LastRevision;    // 提出時に時系列順に付与されるリビジョン

    std::atomic<int> m_SocketDescriptor;    // このパラメータのみ、受理後に設定されるので、 IPC スレッドとディスパッチスレッドのあいだで排他が必要

    bool m_IsTemporaryExcluded;

    AdditionalInfo m_AdditionalInfo;

public:
    UserRequest(
        RequestManager* pRequestManager,
        RequirementPreset requirementPreset,
        ISignalObject* pSignalObject,
        ClientId clientId,
        nn::Bit64 processId) NN_NOEXCEPT;  // RequestManager からのみ呼ぶ
    ~UserRequest() NN_NOEXCEPT;

    nn::Result SetRequirement(const Requirement& requirement) NN_NOEXCEPT;

    nn::Result GetRequirement(Requirement* pOutRequirement) const NN_NOEXCEPT;

    nn::Result SetRequirementPreset(RequirementPreset requirementPreset) NN_NOEXCEPT;

    nn::Result SetRequirementByRevision(uint32_t revision) NN_NOEXCEPT;

    nn::Result SetPriority(uint8_t priority) NN_NOEXCEPT;

    nn::Result SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT;

    nn::Result SetNetworkProfileId(const nn::util::Uuid& uuid) NN_NOEXCEPT;

    nn::Result SetRejectable(bool isRejectableRequest) NN_NOEXCEPT;

    nn::Result SetConnectionConfirmationOption(ConnectionConfirmationOption connectionConfirmationOption) NN_NOEXCEPT;

    nn::Result SetPersistent(bool isPersistent) NN_NOEXCEPT;

    nn::Result SetInstant(bool isInstant) NN_NOEXCEPT;

    nn::Result SetSustainable(bool isSustainable, uint8_t priority) NN_NOEXCEPT;

    nn::Result SetGreedy(bool isGreedy) NN_NOEXCEPT;

    nn::Result SetSharable(bool isSharable) NN_NOEXCEPT;

    nn::Result SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT;

    nn::Result RegisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT;

    nn::Result UnregisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT;

    nn::Result SetClientId(ClientId clientId) NN_NOEXCEPT;

    const Requirement& GetRequirement() const NN_NOEXCEPT;

    // 受理済みであれば Invalid を返す
    ConnectionConfirmationOption GetPresentConnectionConfirmationOption() const NN_NOEXCEPT;

    nn::Result GetResult() const NN_NOEXCEPT;

    RequestState GetRequestState() const NN_NOEXCEPT;

    uint32_t GetRevision() const NN_NOEXCEPT;

    int GetSocketDescriptor() const NN_NOEXCEPT;

    Bit64 GetClientProcessId() const NN_NOEXCEPT;

    ClientId GetClientId() const NN_NOEXCEPT;

    nn::Result GetAdditionalInfo(AdditionalInfo* pOutAdditionalInfo, uint32_t *pOutRevision) const NN_NOEXCEPT;

    // 以下、 RequestManager から呼ぶ関数

    // 要求を受理し、保留状態に遷移しようとします
    nn::Result Receive(uint32_t lastRevision) NN_NOEXCEPT;

    // 要求の受理を拒否します
    nn::Result Refuse(nn::Result result) NN_NOEXCEPT;

    // 要求を却下し、初期状態に遷移しようとします
    // 要求の属性により、ブロック状態や保留状態にとどまることがあります
    nn::Result Reject(nn::Result result) NN_NOEXCEPT;
    nn::Result Reject(nn::Result result, const AdditionalInfo& additionalInfo) NN_NOEXCEPT;

    // 要求を受理し、承認状態に遷移しようとします
    nn::Result Accept() NN_NOEXCEPT;

    //
    bool IsTemporaryExcluded() const NN_NOEXCEPT;
    void ResetTemporaryExcludedFlag() NN_NOEXCEPT;

    // デバッグ用ログ出力
    void Dump() const NN_NOEXCEPT;

private:
    void Signal() NN_NOEXCEPT;
};

}
}
}
