﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/core/nifm_NetworkResource.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_ThreadTypes.h>

#include <functional>

namespace nn
{
namespace nifm
{
namespace detail
{

class RequestManager;
class ConnectionSelector;
class EthernetInterfaceManager;
class NetworkProfileManager;
class PowerStateCoordinator;

class DispatchLoop
{
private:
    static const size_t s_StackSize = 16 * nn::os::ThreadStackAlignment;
    static const int64_t DispatchIntervalInSeconds = 60;

private:
    char m_ThreadStack[s_StackSize + nn::os::ThreadStackAlignment - 1];
    nn::os::ThreadType m_ThreadType;
    static void DispatchLoopThreadFunction(void* pContext) NN_NOEXCEPT;

    nn::os::Event m_TriggerLoopEvent;
    SingleEventHandler m_TriggerLoopEventHandler;

    RequestManager* m_pRequestManager;
    ConnectionSelector* m_pConnectionSelector;
    EthernetInterfaceManager* m_pEthernetInterfaceManager;
    NetworkProfileManager* m_pNetworkProfileManager;
    PowerStateCoordinator* m_pPowerStateCoordinator;

    AggregatedRequestType m_AggregatedRequest;
    TotalNetworkResourceInfo m_TotalNetworkResourceInfo;

    std::function<void() NN_NOEXCEPT> m_DispatchImpl;

    mutable nn::os::SdkMutex m_Mutex;  // 以下の変数の排他を取る Mutex

    bool m_IsPartial;
    bool m_IsToExit;

public:
    DispatchLoop(
        RequestManager* pRequestManager,
        ConnectionSelector* pConnectionSelector,
        EthernetInterfaceManager* pEthernetInterfaceManager,
        NetworkProfileManager* pNetworkProfileManager,
        PowerStateCoordinator* pPowerStateCoordinator
    ) NN_NOEXCEPT;
    ~DispatchLoop() NN_NOEXCEPT;

    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

private:
    void Dispatch() NN_NOEXCEPT;

private:
    void SetToExit(bool isToExit) NN_NOEXCEPT;
    bool IsToExit() NN_NOEXCEPT;
    void TriggerNetworkSwitchIfRequired() NN_NOEXCEPT;

    // 統合リクエストを維持したまま、接続状態を更新して各利用要求に反映します。
    void AdjustRequestsAndConnections() NN_NOEXCEPT;
    // 統合リクエストを更新し、それに合わせて接続状態を更新して各利用要求に反映します。
    // 統合リクエストと接続状態に変化がなければ true を、変化があれば false を返します。
    bool BalanceRequestsAndConnections() NN_NOEXCEPT;

    void RefreshSsidListNetworkProfile() NN_NOEXCEPT;

    void DispatchEthernetInterfaceStandingBy() NN_NOEXCEPT;
    void DispatchFullAwake() NN_NOEXCEPT;
    void DispatchMinimumAwake() NN_NOEXCEPT;
    void DispatchPuttingRequestManagerToSleep() NN_NOEXCEPT;
    void DispatchWaitingForConnectionSelectorsBalance() NN_NOEXCEPT;
    void DispatchPuttingNetworkInterfacesToSleep() NN_NOEXCEPT;
    void DispatchReadyToSleep() NN_NOEXCEPT;
    void DispatchHandlingDisconnectionDuringSleep() NN_NOEXCEPT;

    void DispatchPuttingRequestManagerToShutdown() NN_NOEXCEPT;
    void DispatchWaitingForConnectionFree() NN_NOEXCEPT;
    void DispatchReadyToShutdown() NN_NOEXCEPT;

    static void NotifyNetworkStateToSrepo(
        NetworkType type,
        bool isNetworkResourceStateAvailable,
        bool isInternetAvailabilityConfirmed,
        bool isProhibitedConnectionConfirmationOptionRequestAvailable) NN_NOEXCEPT;
};

}
}
}
