﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileManager.h>
#include <nn/nifm/detail/core/nifm_NetworkResource.h>
#include <nn/nifm/detail/core/nifm_TelemetryContext.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_SignalObject.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>

#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_Optional.h>
#include <nn/os/os_TimerEvent.h>


namespace nn
{
namespace nifm
{
namespace detail
{

// TODO: 有線・無線（インフラ・ローカル）の接続を総合的に表現できるクラスにする
struct ConnectionSettings
{
    int profileIndex;
    const NetworkProfileBase* pNetworkProfile;
    const AccessPointBase* pAccessPoint;
};

class NetworkInterfaceBase;

class ConnectionSelector
{
    NN_DISALLOW_COPY(ConnectionSelector);
    NN_DISALLOW_MOVE(ConnectionSelector);

public:
    static const uint32_t InvalidScanCounter = 0;
    static const int SocketInfoCountMax = 1;

private:
    static const size_t CONNECTION_SETTINGS_LIST_SIZE_MAX = 100;

    static const int64_t VeryLongTimeInSeconds = 100LL * 60 * 60 * 24 * 365;    // 実質的にまわってこず、足してもオーバーフローしないくらいの長い時間
#if NN_DETAIL_NIFM_CONFIG_ENABLE_AUTO_SCAN
    static const int64_t ScanIntervalInSeconds = 60;                        // 自動スキャンの走る間隔
#else
    static const int64_t ScanIntervalInSeconds = VeryLongTimeInSeconds;     // 自動スキャンの走る間隔
#endif
    static const int64_t ScanDataLifeTimeInSeconds = 60;                    // スキャンデータをクリアするまでの時間

private:
    nn::util::IntrusiveList<NetworkInterfaceBase, nn::util::IntrusiveListBaseNodeTraits<NetworkInterfaceBase>> m_NetworkInterfaceList;

    mutable nn::os::SdkRecursiveMutex m_Mutex;

    uint32_t m_ScanCounter;
    nn::Result m_ScanResult;
    SignalList m_ScanSignalList;
    nn::TimeSpan m_ScanInterval;
    nn::os::TimerEvent m_ScanTimerEvent;
    SingleTimerEventHandler m_ScanTimerEventHandler;

    class ScanTimerEventCallback : public CallbackObject
    {
    private:
        ConnectionSelector& m_ConnectionSelector;

    public:
        explicit ScanTimerEventCallback(ConnectionSelector& connectionSelector)
            : m_ConnectionSelector(connectionSelector)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            m_ConnectionSelector.Scan();
        }
    } m_ScanTimerEventCallback;

    AccessPointList<FrameHeap<20 * 1024>> m_AccessPointList;    // 取りこぼさないように、すくなくとも無線スキャンのバッファより大きくなければならない
    NetworkProfileManager* m_pNetworkProfileManager;

    MultiEventHandler m_ConnectionEventHandler;

    struct ConnectionSettingList
    {
        static const int CountMax = 100;

        int count;
        ConnectionSettings buffer[CountMax];
        ConnectionSettings* ptrList[CountMax];
    };

    bool m_IsAwake;

    TelemetryContext m_TelemetryContext;

private:
    nn::Result ConfirmScanExecutable() const NN_NOEXCEPT;
    nn::Result ScanCore() NN_NOEXCEPT;

    // 現在の接続そのままで新しい統合リクエストを受け入れられるか確認し、
    // 受け入れられれば保持している統合リクエストを更新します
    nn::Result ConfirmKeepingCurrentConnection(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // 指定した時間以内におこなったスキャン結果を取得します
    void GetLatestAccessPointList(nn::TimeSpan timeSpan) NN_NOEXCEPT;

    // アクセスポイントのリストを更新します
    // 何らかの更新があった場合 true を返します
    bool UpdateAccessPointList() NN_NOEXCEPT;

    // 接続の候補を列挙する
    // TODO: ConnectionSettings の名前を変える
    void EnumerateConnectionSettingsList(
        ConnectionSettingList* pOutConnectionSettingList,
        const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT;

    // 片っ端から接続試行して、最初に接続に成功した設定を返す
    nn::Result ConnectAny(
        nn::util::optional<AdditionalInfo>* pOutAdditionalInfo,
        ConnectionSettings* pOutConnectionSettings,
        nn::util::optional<IpAddressSetting>* pOutIpAddressSetting,
        nn::util::optional<DnsSetting>* pOutDnsSetting,
        const ConnectionSettingList& connectionSettingList,
        const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT;

    nn::Result GetRedirectInfo(AdditionalInfo* pOutAdditionalInfo) const NN_NOEXCEPT;

    // テレメトリ情報更新の必要性を調べる
    bool IsTelemetryUpdateNeeded(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // テレメトリ情報更新
    void UpdateTelemetryContext(
        Result result,
        const nn::util::optional<NetworkProfileData>& networkProfile,
        const nn::util::optional<AccessPointData>& accessPoint,
        const nn::util::optional<IpAddressSetting>& ipAddressSetting,
        const nn::util::optional<DnsSetting>& dnsSetting,
        const nn::util::optional<AdditionalInfo>& additionalInfo) NN_NOEXCEPT;

public:
    explicit ConnectionSelector(NetworkProfileManager *pNetworkProfileManager) NN_NOEXCEPT;
    virtual ~ConnectionSelector() NN_NOEXCEPT;

    nn::Result Scan() NN_NOEXCEPT;

    // スキャン間隔を指定します。
    // 指定時点から再度計測しなおします。
    void SetScanInterval(const nn::TimeSpan& timeSpan) NN_NOEXCEPT;

    EventHandler& GetScanTimerEventHandler() NN_NOEXCEPT
    {
        return m_ScanTimerEventHandler;
    }

    void TriggerScan(uint32_t* pOutScanCounter, SignalListItem& signalListItem) NN_NOEXCEPT;    // スキャンを依頼する
    uint32_t GetScanCounter() const NN_NOEXCEPT;
    nn::Result GetScanResult() const NN_NOEXCEPT;
    void RemoveScanCompleteListener(SignalListItem& signalListItem) NN_NOEXCEPT;

    nn::Result Register(NetworkInterfaceBase* pNetworkInterfaceInterface) NN_NOEXCEPT;

    nn::Result Unregister(NetworkInterfaceBase* pNetworkInterfaceInterface) NN_NOEXCEPT;

    nn::Result Clear() NN_NOEXCEPT;

    // aggregatedRequestType に対して見合う接続状態への切り替えを試行します
    // aggregatedRequestType を満たす接続状態になった場合、成功を返します
    // aggregatedRequestType を満たせなかった場合、必要であれば現在の接続を切断し、失敗を返します
    // この関数を抜けたら、 RequestManager::NotifyTotalNetworkResource を呼んで要求の状態を最新に反映したうえ、
    // ReleaseNetworkResourceLost を呼んで Lost 状態を Release します。
    // TODO: [TORIAEZU] m_AccessPointList の内容へのアクセスをロックを取らずにおこなうため、
    //                  これの操作をおこなう Update() とはスレッドセーフではありません
    nn::Result Renew(AdditionalInfo *pOutAdditionalInfo, const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT;

    nn::Result GetCurrentNetworkProfile(NetworkProfileData *pOutNetworkProfileData) NN_NOEXCEPT;

    nn::Result GetTotalNetworkResourceInfo(TotalNetworkResourceInfo* pTotalNetworkResourceInfo) const NN_NOEXCEPT;

    nn::Result ReleaseNetworkResourceLost(const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT;

    EventHandler& GetConnectionEventHandler() NN_NOEXCEPT
    {
        return m_ConnectionEventHandler;
    }

    nn::Result EnumerateNetworkInterfaces( NetworkInterfaceInfo *pNetworkInterfaceInfo, int* pOutCount, int inCount ) const NN_NOEXCEPT;


    nn::Result GetScanData(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT;

    nn::Result GetCurrentIpAddress( IpV4Address* pOutIpAddress ) const NN_NOEXCEPT;

    nn::Result GetCurrentIpConfigInfo( IpAddressSetting* pOutIpAddressSetting, DnsSetting* pOutDnsSetting ) const NN_NOEXCEPT;

    nn::Result GetCurrentAccessPoint( AccessPointData* pOutAccessPointData ) NN_NOEXCEPT;

    // 無線接続中かつ有線がリンクアップしている、という条件を満たしているか否かを返す
    bool IsNicSwitchingRequired() const NN_NOEXCEPT;

    // すべての NIC が Free であることを保証する
    // Free でない NIC があれば Disconnect し、失敗を返す
    nn::Result EnsureConnectionFree() NN_NOEXCEPT;

    // 接続設定の変更に応じた処理
    void HandleNetworkProfileModification() NN_NOEXCEPT;

    // テレメトリ情報が更新された際にシグナルされるイベントのハンドル取得
    nn::os::NativeHandle GetTelemetrySystemEventReadableHandle() NN_NOEXCEPT;

    // テレメトリ情報取得
    void GetTelemetryInfo(TelemetryInfo* pOutTelemetryInfo) NN_NOEXCEPT;

    nn::Result ConfirmInterfaceHealth() const NN_NOEXCEPT;

    nn::Result SetTcpSessionInformation(const SocketInfo socketInfoArray[], int count) NN_NOEXCEPT;

    // 登録済みのネットワークインタフェースに対してスキャン対象チャンネルを設定します。
    nn::Result SetScanChannels(const int16_t scanChannelArray[], int count) NN_NOEXCEPT;

    // 指定されたチャンネルに対するスキャンが可能か否かを判定します。
    bool IsScanAllowed(const int16_t scanChannelArray[], int count) NN_NOEXCEPT;

    nn::Result PutToSleep() NN_NOEXCEPT;
    nn::Result WakeUp() NN_NOEXCEPT;
    //bool IsReadyToSleep() const NN_NOEXCEPT;
};

}
}
}

