﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.win32.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_Heap.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class EthernetInterfaceManagerBase
{
    NN_DISALLOW_COPY(EthernetInterfaceManagerBase);
    NN_DISALLOW_MOVE(EthernetInterfaceManagerBase);

private:
    static const int EthernetInterfaceCountMax = 1;     // 同時に扱える USB Ether アダプターの最大数

private:
    mutable nn::os::SdkMutex m_Mutex;

    EthernetInterface m_EthernetInterface;  // Windows では固定で1つ

    nn::os::Event m_InterfaceGroupEvent;    // m_InterfaceGroupEventHandler の初期化のために存在するがシグナルしない
    class InterfaceGroupEventCallback : public CallbackObject
    {
    private:
        EthernetInterfaceManagerBase* m_pEthernetInterfaceManagerBase;

    public:
        explicit InterfaceGroupEventCallback(EthernetInterfaceManagerBase* pEthernetInterfaceManagerBase)
            : m_pEthernetInterfaceManagerBase(pEthernetInterfaceManagerBase)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE;
    } m_InterfaceGroupEventCallback;
    SingleEventHandler m_InterfaceGroupEventHandler;

    bool m_IsEnabled;
    bool m_IsAppended;  // 主にデバッグ用状態管理に利用

public:
    // 現在の環境に接続された有線 NIC の MAC アドレスを列挙します。
    // Windows 環境ではホスト PC に接続されたすべての有線 NIC が対象になるため、
    // ターゲットデバイスとして利用できるより多くの NIC が列挙されることがあります。
    // pOutCount はすべての有線 NIC の数を返し、 inCount の値より大きくなることがあります。
    void GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) const NN_NOEXCEPT;

public:
    EthernetInterfaceManagerBase() NN_NOEXCEPT;

    virtual ~EthernetInterfaceManagerBase() NN_NOEXCEPT;

    // USB Ether アダプターの挿抜イベントハンドラへの参照を取得します
    // これのコールバックを呼ぶことで、派生クラスで実装したコールバックが実行されます
    // ただし、 Windows 版では起動後に1回発火するのみです
    EventHandler& GetPluggingEventHandler() NN_NOEXCEPT
    {
        return m_InterfaceGroupEventHandler;
    }

    virtual void AppendCallback(EthernetInterface& ethernetInterface) NN_NOEXCEPT = 0;
    virtual void RemoveCallback(EthernetInterface& ethernetInterface) NN_NOEXCEPT = 0;

    nn::Result SetEnabled(bool isEnabled) NN_NOEXCEPT;
    bool IsEnabled() const NN_NOEXCEPT;

    // Ethernet インターフェースの挿抜状態を最新に反映させます
    // 新しい Ethernet インターフェースを発見した場合は true を返します
    bool Renew() NN_NOEXCEPT;

    // pMultiWaitHolder が指す多重待ちオブジェクトホルダーを保持するインターフェースを探し、
    // 見つかればそのインターフェースへのポインタを、見つからなければ nullptr を返します。
    // この関数の呼び出し中、および取り出したインターフェースへの操作中には
    // インターフェースの増減が発生するような操作をおこなってはいけません。
    EthernetInterface* Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;

    // スリープ明け直後の、 USB Ethernet アダプターの検出を待っている期間か
    bool IsStandingBy() NN_NOEXCEPT;

    nn::Result PutToSleep() NN_NOEXCEPT;
    nn::Result WakeUp() NN_NOEXCEPT;

private:
    void InterfaceGroupEventCallback() NN_NOEXCEPT;

protected:
    // 派生クラスのデストラクタから呼んでください
    // 内部で RemoveCallback を呼びます
    void RemoveAll() NN_NOEXCEPT;
};

}
}
}

