﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/os/os_Thread.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/nfc/nfc_Types.h>
#include <nn/nfc/nfc_Result.h>
#include <nn/nfc/nfc_PrivateResult.h>
#include <nn/nfc/nfc_NfpResult.internal.h>
#include <nn/nfc/server/util/nfc_ThreadStack.h>
#include <nn/nfc/server/nfc_Device.h>
#include <nn/nfp/server/nfp_Manager.h>
#include <nn/nfc/mifare/server/nfc_MifareManager.h>
#include <nn/nfc/pt/server/nfc_PtManager.h>

#define NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(manager, service, deviceHandle, access) \
    nn::Result __result;                                                \
    nn::nfc::server::Device* __device;                                  \
    nn::xcd::DeviceHandle __xcdDeviceHandle;                            \
    NN_RESULT_DO(manager->BeforeAccessDevice(&__device, &__xcdDeviceHandle, service, deviceHandle)); \
    __result = __device->access;                                        \
    return manager->AfterAccessDevice(service, __device, __xcdDeviceHandle, __result)

namespace nn { namespace nfc { namespace server {

    class Manager
    {
        friend nn::nfp::server::Manager;
        friend nn::nfc::mifare::server::Manager;
        friend nn::nfc::pt::server::Manager;

    public:
        nn::Result Initialize(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        void Finalize(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        nn::Result ListDevices(nn::nfc::DeviceHandle* pOutBuffer, int* pOutCount, nn::nfc::server::core::Service* service, int bufferCount, bool forApi) NN_NOEXCEPT;
        nn::Result StartDetection(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT;
        nn::Result StopDetection(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetTagInfo(nn::nfc::TagInfo* pOutTagInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        State GetState(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        nn::nfc::server::Device::State GetDeviceState(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetActivateEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetDeactivateEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetAvailabilityChangeEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service) NN_NOEXCEPT;

        nn::Result InitializeSystem(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        void FinalizeSystem(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;

        //nn::nfp
        nn::Result Mount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfp::ModelType modelType, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT;
        nn::Result Unmount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result OpenApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT;
        nn::Result GetApplicationArea(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT;
        nn::Result SetApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize) NN_NOEXCEPT;
        nn::Result RecreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT;
        nn::Result Flush(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result Restore(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT;
        nn::Result GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;

        nn::Result Format(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result SetRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT;
        nn::Result DeleteRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result DeleteApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result ExistsApplicationArea(bool* outValue, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result GetAll(nn::nfp::NfpData* pOutNfpData, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result SetAll(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::NfpData& nfpData) NN_NOEXCEPT;
        nn::Result FlushDebug(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result BreakTag(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfp::BreakType breakType) NN_NOEXCEPT;
        nn::Result ReadBackupData(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, size_t bufferSize) NN_NOEXCEPT;
        nn::Result WriteBackupData(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize) NN_NOEXCEPT;
        nn::Result WriteNtf(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize, nn::nfp::NtfWriteType ntfWriteType) NN_NOEXCEPT;
        size_t GetApplicationAreaSize() NN_NOEXCEPT;

        //nn::nfc::mifare
        nn::Result ReadMifare(nn::nfc::MifareReadBlockData* pOutBlockData, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::MifareReadBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT;
        nn::Result WriteMifare(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::MifareWriteBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT;

        //nn::nfc::pt
        nn::Result SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT;
        nn::Result KeepPassThroughSession(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result ReleasePassThroughSession(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;

    private:
        Manager() NN_NOEXCEPT;
        ~Manager() NN_NOEXCEPT;
        nn::Result InitializePrivate(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        void FinalizePrivate(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        void SetState(State state) NN_NOEXCEPT;
        nn::Result GetDevice(Device** pOutDevice, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, bool check) NN_NOEXCEPT;
        nn::Result GetDevice(Device** pOutDevice, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        static void LinkMonitoringThread(void* arg) NN_NOEXCEPT;
        nn::Result UpdateDevices() NN_NOEXCEPT;
        bool IsInitialized() NN_NOEXCEPT;
        nn::Result CheckService(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        nn::Result CheckNfcEnabled(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
        nn::Result CheckAwake() NN_NOEXCEPT;
        nn::Result BeforeAccessDevice(Device** pOutDevice, nn::xcd::DeviceHandle* pOutXcdDeviceHandle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT;
        nn::Result AfterAccessDevice(nn::nfc::server::core::Service* service, Device* device, nn::xcd::DeviceHandle xcdDeviceHandle, nn::Result result) NN_NOEXCEPT;

    public:
        static Manager& GetInstance() NN_NOEXCEPT
        {
            static Manager instance;
            return instance;
        }

    private:
        nn::nfc::State m_State;
        std::unique_ptr<Device> m_DeviceList[nn::nfc::DeviceCountMax];

        nn::os::SystemEventType m_LinkEvent;
        nn::os::MultiWaitHolderType m_LinkHolder;
        nn::os::MultiWaitType m_LinkMonitoringMultiWait;
        nn::os::SystemEventType m_TerminateLinkMonitoringEvent;
        nn::os::MultiWaitHolderType m_TerminateLinkMonitoringHolder;
        nn::os::ThreadType m_LinkMonitoringThread;
        std::unique_ptr<nn::nfc::server::util::ThreadStack> m_LinkMonitoringThreadStack;

        mutable nn::os::ReaderWriterLock m_ReaderWriterLock;
        bool m_IsRunningThread;
        int m_InitializeCount;

        nn::nfp::server::Manager m_Nfp;
        nn::nfc::mifare::server::Manager m_NfcMifare;
        nn::nfc::pt::server::Manager m_NfcPt;
    };

}}}  // namespace nn::nfc::server
