﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/news_TypesAdmin.h>
#include <nn/nn_ApplicationId.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace news {

/*!
    @name システム設定
    @{
*/

/*!
    @brief      ニュースデータの復号鍵を生成するパスフレーズを設定します。

    @param[in]  appId       アプリケーション ID。
    @param[in]  passphrase  パスフレーズ。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()
        - passphrase != nullptr

    @details
                ニュースデータの復号鍵は、アプリケーション ID とパスフレーズの組み合わせによって生成されます。

                本関数で設定したパスフレーズは、システム領域に保存されます。@n
                そのため、本関数はシステムの初回起動時に 1 度だけ呼び出してください。

                パスフレーズがすでに保存されていた場合、本関数は新しいパスフレーズで上書き保存します。
*/
nn::Result SetPassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT;

/*!
    @}
    @name 購読設定
    @{
*/

/*!
    @brief      ニュースの購読状況を設定します。

    @param[in]  topicId トピック ID。
    @param[in]  status  購読状況。

    @return     処理結果。

    @pre
        - topicId != nullptr

    @details
                本関数は廃止予定です。@n
                代わりに、 @ref SetSubscriptionStatus を使用してください。
*/
NN_DEPRECATED nn::Result SetSubscribeStatus(const char* topicId, SubscribeStatus status) NN_NOEXCEPT;

/*!
    @brief      ニュースの購読状態を設定します。

    @param[in]  topicId トピック ID。
    @param[in]  status  購読状態。

    @return     処理結果。

    @pre
        - topicId != nullptr
*/
nn::Result SetSubscriptionStatus(const char* topicId, SubscriptionStatus status) NN_NOEXCEPT;

/*!
    @brief      ニュースの購読状況を取得します。

    @param[out] outStatus   購読状況。
    @param[in]  topicId     トピック ID。

    @return     処理結果。

    @pre
        - outStatus != nullptr
        - topicId != nullptr

    @details
                本関数は廃止予定です。@n
                代わりに、 @ref GetSubscriptionStatus を使用してください。
*/
NN_DEPRECATED nn::Result GetSubscribeStatus(SubscribeStatus* outStatus, const char* topicId) NN_NOEXCEPT;

/*!
    @brief      ニュースの購読状態を取得します。

    @param[out] outStatus   購読状態。
    @param[in]  topicId     トピック ID。

    @return     処理結果。

    @pre
        - outStatus != nullptr
        - topicId != nullptr
*/
nn::Result GetSubscriptionStatus(SubscriptionStatus* outStatus, const char* topicId) NN_NOEXCEPT;

/*!
    @brief      トピックリストを取得します。

    @param[out] outCount    取得したトピック ID 数。
    @param[out] outTopicIds トピック ID リスト。
    @param[in]  count       トピック ID リストの要素数。
    @param[in]  filter      フィルター。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - outTopicIds != nullptr
        - count > 0

    @details
                トピックリストの最大数は @ref TopicCountMax になります。
*/
nn::Result GetTopicList(int* outCount, TopicId* outTopicIds, int count, SubscriptionStatusFilter filter) NN_NOEXCEPT;

/*!
    @brief      ニュースの自動購読を要求します。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                本関数は、システムの自動購読管理リストにアプリケーションを登録します。

                自動購読管理リストにアプリケーションが登録された場合、アプリケーション ID を引数に購読可能なトピックリストを非同期で取得します。@n
                ローカルのトピックリストに存在しないトピックが発見された場合、システムは自動的にそれを購読します。

                ユーザーによって購読解除されたトピックを再度自動購読することはありません。

                自動購読管理リストは電源断によって消失します。
*/
nn::Result RequestAutoSubscription(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @}
    @name   本体初期化
    @{
*/

/*!
    @brief      ニュースデータの保存ストレージをクリアします。

    @return     処理結果。

    @details
                ニュースデータ、ニュース DB、および、ニュースの受信履歴をすべて削除します。
*/
nn::Result ClearStorage() NN_NOEXCEPT;

/*!
    @brief      ニュースの購読状態をすべて未設定状態にします。

    @return     処理結果。

    @details
                本関数は廃止予定です。@n
                代わりに、 @ref ClearSubscriptionStatusAll を使用してください。
*/
NN_DEPRECATED nn::Result ClearSubscribeStatusAll() NN_NOEXCEPT;

/*!
    @brief      ニュースの購読状態をすべて未設定状態にします。

    @return     処理結果。
*/
nn::Result ClearSubscriptionStatusAll() NN_NOEXCEPT;

/*!
    @}
    @name システム更新
    @{
*/

/*!
    @brief      システム更新が要求されているかどうかを確認します。

    @param[out] outIsRequired   システム更新が要求されているかどうか。

    @return     処理結果。

    @pre
        - outIsRequired != nullptr

    @details
                ニュースデータは仕様のリッチ化によって、項目の追加や列挙値の取りうる範囲が拡張されます。@n
                しかし、サーバーで配信するニュースデータのバージョン更新とそれを解釈するためのシステム更新は、同時に行われる保証はありません。

                システムは、解釈可能ではないニュースデータの配信を検知した場合、システム更新が必要と判断します。@n
                システム更新によって新しいニュースデータが解釈可能になることが期待できるため、ユーザーに本体更新を促すことを推奨します。
*/
nn::Result IsSystemUpdateRequired(bool* outIsRequired) NN_NOEXCEPT;

/*!
    @}
    @name   受信要求
    @{
*/

/*!
    @brief      ニュースの即時受信を要求します。

    @param[in]  topicId トピック ID。

    @return     処理結果。

    @pre
        - topicId != nullptr

    @details
                プッシュ通知による受信要求を、任意のタイミングで発生させることができます。
*/
nn::Result RequestImmediateReception(const char* topicId) NN_NOEXCEPT;

/*!
    @}
    @name ローカルニュース
    @{
*/

/*!
    @brief      任意のローカルニュースを投稿します。

    @param[in]  data        ニュースデータ。
    @param[in]  dataSize    ニュースデータのサイズ。

    @return     処理結果。

    @pre
        - data != nullptr
        - dataSize > 0

    @details
                本関数は、ニュース配信サーバーで生成されるニュースデータフォーマットに準拠したニュースデータを投稿します。@n
                既に同じニュースが投稿されていた場合、本関数は失敗します。

                本関数は、ユーザーアカウントの同意なしにニュースを投稿します。
*/
nn::Result PostLocalNews(const void* data, size_t dataSize) NN_NOEXCEPT;

/*!
    @}
    @name ニュースデータ復号
    @{
*/

/*!
    @brief      暗号化されたアーカイブファイル（ニュースデータ）を復号します。

    @param[out] outSize     復号済みファイルのサイズ。
    @param[in]  decoded     復号済みファイルを格納するバッファ。
    @param[in]  decodedSize 復号済みファイルを格納するバッファのサイズ。
    @param[in]  encoded     暗号化されたアーカイブファイル。
    @param[in]  encodedSize 暗号化されたアーカイブファイルのサイズ。

    @return     処理結果。
    @retval     ResultPassphraseNotSaved        パスフレーズが保存されていない。
    @retval     ResultDataVerificationFailed    データの検証に失敗した。
    @retval     ResultUnsupportedFormatDetected サポートしていないフォーマットのデータを検出した。

    @pre
        - outSize != nullptr
        - decoded != nullptr
        - decodedSize > 0
        - encoded != nullptr
        - encodedSize > 0

    @details
                復号済みファイルを格納するバッファのサイズは、暗号化されたアーカイブファイルのサイズと同じにすることを推奨します。
*/
nn::Result DecodeArchiveFile(size_t* outSize, void* decoded, size_t decodedSize,
    const void* encoded, size_t encodedSize) NN_NOEXCEPT;

/*!
    @}
    @name イベント
    @{
*/

/*!
    @brief      ニュースの新着イベントを登録します。

    @param[out] outEvent    新着イベント。

    @return     処理結果。

    @pre
        - outEvent != nullptr

    @details
                バックグラウンドでニュースを新規受信した際に、登録されたイベントが Signal されます。
*/
nn::Result RegisterNewlyArrivedEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT;

/*!
    @brief      ニュースの上書きイベントを登録します。

    @param[out] outEvent    上書きイベント。

    @return     処理結果。

    @pre
        - outEvent != nullptr

    @details
                バックグラウンドでニュースを上書き受信した際に、登録されたイベントが Signal されます。
*/
nn::Result RegisterOverwriteEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT;

/*!
    @}
*/

}}
