﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>
#include <sqlite3.h>

namespace nn { namespace news { namespace detail { namespace service { namespace util {

/*!
    @brief      SQLite のユーティリティです。
*/
class Sqlite
{
public:
    /*!
        @brief      ユーティリティが扱うエスケープ済み文字列の最大サイズです。
    */
    static const size_t EscapedStringSize = 64;

public:
    /*!
        @brief      アロケーターのセットアップを行います。

        @return     処理結果。
    */
    static nn::Result SetupAllocator() NN_NOEXCEPT;

    /*!
        @brief      グローバルヒープを利用するアロケーターのセットアップを行います。

        @return     処理結果。
    */
    static nn::Result SetupGlobalHeapAllocator() NN_NOEXCEPT;

    /*!
        @brief      パフォーマンスを向上させるパラメーターを設定します。

        @param[in]  pHandle DB ハンドル。

        @pre
            - pHandle != nullptr

        @return     処理結果。
    */
    static nn::Result SetPerformanceImprovementParameters(sqlite3* pHandle) NN_NOEXCEPT;

    /*!
        @brief      データベース用のカスタムクエリ関数を登録します。

        @param[in]  pHandle DB ハンドル。

        @pre
            - pHandle != nullptr

        @return     処理結果。

        @details
                    本関数を呼び出すことで、以下のカスタムクエリ関数が利用可能になります。

                    - N_IF
                    - N_SWITCH
                    - N_RAND
                    - N_PMATCH
    */
    static nn::Result RegisterCustomFunctions(sqlite3* pHandle) NN_NOEXCEPT;

    /*!
        @brief      データベースのページサイズを取得します。

        @param[out] outPageSize ページサイズ。
        @param[in]  pHandle     DB ハンドル。

        @pre
            - outPageSize != nullptr
            - pHandle != nullptr

        @return     処理結果。
    */
    static nn::Result GetPageSize(int* outPageSize, sqlite3* pHandle) NN_NOEXCEPT;

    /*!
        @brief      テーブルに整数型のカラムを追加します。

        @param[in]  pHandle         DB ハンドル。
        @param[in]  tableName       テーブル名。
        @param[in]  columnName      カラム名。
        @param[in]  defaultValue    デフォルト値。

        @return     処理結果。
        @retval     ResultSqliteErrorError  同名のカラムが既に存在しています。(SQLITE_ERROR)

        @pre
            - pHandle != nullptr
            - tableName != nullptr
            - columnName != nullptr
    */
    static nn::Result AddColumn(sqlite3* pHandle,
        const char* tableName, const char* columnName, int64_t defaultValue) NN_NOEXCEPT;

    /*!
        @brief      テーブルに文字列型のカラムを追加します。

        @param[in]  pHandle         DB ハンドル。
        @param[in]  tableName       テーブル名。
        @param[in]  columnName      カラム名。
        @param[in]  defaultValue    デフォルト値。

        @return     処理結果。
        @retval     ResultSqliteErrorError  同名のカラムが既に存在しています。(SQLITE_ERROR)

        @pre
            - pHandle != nullptr
            - tableName != nullptr
            - columnName != nullptr
            - defaultValue != nullptr
    */
    static nn::Result AddColumn(sqlite3* pHandle,
        const char* tableName, const char* columnName, const char* defaultValue) NN_NOEXCEPT;

    /*!
        @brief      SQLite の処理結果を nn::Result に変換します。

        @param[in]  result  処理結果（SQLITE_XXXX）。

        @return     処理結果。
    */
    static nn::Result HandleError(int result) NN_NOEXCEPT;

    /*!
        @brief      文字列をエスケープします。

        @param[out] output          エスケープ後の文字列を格納するバッファ。
        @param[in]  outputSize      エスケープ後の文字列を格納するバッファのサイズ。
        @param[in]  input           入力文字列。
        @param[in]  isSingleQuote   文字列の囲みをシングルクォートで表現するかどうか。

        @pre
            - output != nullptr
            - outputSize > 0
            - input != nullptr

        @return     正しくエスケープできたかどうか。

        @details
                    本関数は、以下の場合失敗します。

                    - 入力文字列が妥当な UTF-8 文字列でない。
                    - バッファが不足している。
    */
    static bool EscapeString(char* output, size_t outputSize, const char* input, bool isSingleQuote = true) NN_NOEXCEPT;
};

#define SQLITE_DO(exp) \
    do                                                                  \
    {                                                                   \
        int result_ = (exp);                                            \
                                                                        \
        if (!(result_ == SQLITE_OK || result_ == SQLITE_DONE))          \
        {                                                               \
            return detail::service::util::Sqlite::HandleError(result_); \
        }                                                               \
    }                                                                   \
    while (NN_STATIC_CONDITION(false))

}}}}}
