﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Web 認証アプレットの起動時引数の受け取りをおこなうクラスの非公開ヘッダ
 */

#pragma once

#include <nn/la/la_NifmToWifiWebAuthArguments.h>
#include <nn/ae/ae.h>


namespace nn { namespace la {

/**
 * @brief   NIFM から渡される、Web 認証アプレットの起動時引数の受け取りをおこなうクラスです。
 *          これを利用するアプレットは、 libnn_applet.a, libnn_la_callee.a, libnn_ae.a をリンクする必要があります。
 *
 * @details
 *  アプレットは、このクラスによって受け取られた引数を利用して、ネットワーク利用要求を適切に設定します。
 *  その際は、 nn::nifm::SetRequestRequirementPreset() によって nn::nifm::RequirementPreset_InternetForWifiWebAuthApplet を設定した後に、各引数を設定してください。
 */
class NifmToWifiWebAuthArgumentsReader
{
private:
    bool m_IsInitialized;

    NN_DETAIL_LA_CURRENT_NIFMTOWIFIWEBAUTH_ARGUMENTS_DATA_TYPE m_NifmToWifiWebAuthArgumentsData;

public:
    /**
     * @brief   コンストラクタです。
     *
     * @details
     */
    NifmToWifiWebAuthArgumentsReader() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     *
     * @details
     */
    ~NifmToWifiWebAuthArgumentsReader() NN_NOEXCEPT;


    /**
     * @brief   入力チャンネルから共通引数の pop を試みます。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *  取得に失敗した場合でも、入力チャンネルの状態に副作用を与えることがあります。
     *  内部で ReadFromStorage も呼び出します。
     */
    bool TryPopFromInChannel() NN_NOEXCEPT;

    /**
     * @brief   入力ストレージから実際にデータを読みます。
     *
     * @param[in] storageHandle 共通データを読み出すストレージのハンドルです。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *    Storage の Pop を外部でおこないたい場合にはこちらを使用します。
     */
    bool ReadFromStorage(nn::ae::StorageHandle storageHandle) NN_NOEXCEPT;

    /**
     * @brief   認証のためにアクセスする必要がある URL を取得します。
     *
     * @return  認証のためにアクセスする必要がある URL 文字列のポインタが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    const char* GetAuthUrlPointer() const NN_NOEXCEPT;

    /**
     * @brief   疎通確認のためにアクセスする必要がある URL を取得します。
     *
     * @return  疎通確認のためにアクセスする必要がある URL 文字列のポインタが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    const char* GetConnTestUrlPointer() const NN_NOEXCEPT;

    /**
     * @brief   利用要求提出時に必要とされる接続設定 ID を取得します。
     *
     * @return  利用要求提出時に必要とされる接続設定 ID の参照が返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     *  nn::nifm::SetRequestNetworkProfileId に与えることで、認証が必要とされた接続設定を利用要求に設定することができます。
     *
     */
    const nn::util::Uuid& GetProfileId() const NN_NOEXCEPT;

    /**
     * @brief   利用要求提出番号を取得します。
     *
     * @return  利用要求提出番号が返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     *  取得された提出番号 を nn::nifm::SetRequestRequirementBySubmitId() に与えることで、元の利用要求と同等の要件を設定することができます。
     */
    uint32_t GetSubmitId() const NN_NOEXCEPT;
};

}} // namespace nn::la

