﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレット起動に関する非公開ヘッダ
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/applet/applet.h>
#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/la/la_Result.h>
#include <cstring>

/**
 * @namespace   nn::la
 *
 * @brief   ライブラリアプレット呼び出しの名前空間です。
 *
 * @details ライブラリアプレットを呼び出すための、shim ライブラリの名前空間です。
 */

namespace nn { namespace la {

/**
 * @brief ライブラリアプレット起動時のフック関数への引数
 */
struct LibraryAppletCreateHookArg
{
    nn::applet::AppletId            appletId;          //!< アプレットIDです。
    nn::applet::LibraryAppletMode   libraryAppletMode; //!< アプレットモードです。
    bool                            isExtremity;       //!< 起動しようとしている LA が末端ならば true になります。

    /**
    * @brief デフォルト値で初期化します。
    *
    * @param[in] appletId_     アプレットIDです。
    * @param[in] libraryAppletMode_ アプレットの起動モードです。
    * @param[in] isExtremity_ 起動しようとしている LA が末端ならば true になります。
    */
    void Initialize(nn::applet::AppletId appletId_, nn::applet::LibraryAppletMode libraryAppletMode_, bool isExtremity_ ) NN_NOEXCEPT;
};

/**
 * @brief ライブラリアプレット起動時のフック関数の返り値型
 */
enum LibraryAppletCreateHookResult
{
    LibraryAppletCreateHookResult_Accept, //!< ライブラリアプレットの起動を許可します
    LibraryAppletCreateHookResult_Reject  //!< ライブラリアプレットの起動をリジェクトします
};

//------------------------------------------------------------------------
/**
 * @brief ライブラリアプレット起動時のフック関数の型です。
 *
 * @param[in] arg アプレットの起動パラメータです。
 * @result    アプレットを起動するかどうかを返します。
 */
typedef LibraryAppletCreateHookResult (*LibraryAppletCreateHook)(const LibraryAppletCreateHookArg& arg );

//------------------------------------------------------------------------
/**
 * @brief ライブラリアプレット起動時のフックを設定します。
 *
 * @details
 *   nn::applet::CreateApplet の直後に呼び出されるコールバックです。
 */
void SetLibraryAppletCreateHook(LibraryAppletCreateHook hook) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @brief ライブラリアプレット起動時のフック関数を呼び出します。
 *
 * @param[in] arg アプレットの起動パラメータです。
 * @result    アプレットを起動するかどうかを返します。
 */
LibraryAppletCreateHookResult InvokeLibraryAppletCreateHook(const LibraryAppletCreateHookArg& arg) NN_NOEXCEPT;


//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief StartLibraryAppletに渡す構造体です。
 *
 * @details
 *  この情報が呼び出し元LAに渡され、ProgramWindingするかどうかなどの判定に利用されます。
 *
 */
struct LibraryAppletStartHookUserArg
{
    //! 起動しようとしているLAが末端LAならばtrueに設定してください。
    bool isExtremity;
};

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief ライブラリアプレットの InChannel にストレージを送ります。
 *
 * @param[in]  laHandle       ライブラリアプレットのハンドル
 * @param[in]  buffer         起動パラメータ格納用バッファ
 * @param[in]  bufferSize     起動パラメータのバッファサイズ
 */
void PushToInChannel( nn::applet::LibraryAppletHandle laHandle, const void* buffer, size_t bufferSize ) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief ライブラリアプレットの OutChannel からデータを取得します。
 *
 * @param[in]   laHandle      ライブラリアプレットのハンドル
 * @param[out]  pBuffer       パラメータ取得用バッファ
 * @param[in]   bufferSize    パラメータ取得用のバッファサイズ
 * @param[out]  pReadSize     取得したデータサイズ
 */
void PopFromOutChannel( nn::applet::LibraryAppletHandle laHandle, void* pBuffer, size_t bufferSize, size_t* pReadSize = nullptr ) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief ライブラリアプレットを生成するための、簡易 API。サンプル実装です。
 *
 * @param[out]  laHandle                          ライブラリアプレットハンドルです。
 * @param[in]   appletId 作成したいライブラリアプレットのアプレット ID を指定します。
 * @param[in]   libraryAppletMode ライブラリアプレットモードを指定します。
 * @param[in]   isExtremity                       起動するのが末端の LA であれば true。そうでなければ false にする。
 */
nn::Result CreateLibraryAppletEasy(
    nn::applet::LibraryAppletHandle* pOut,
    nn::applet::AppletId appletId,
    nn::applet::LibraryAppletMode libraryAppletMode,
    bool isExtremity ) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief ライブラリアプレットをFG起動します。
 *
 * @param[in]  laHandle                          ライブラリアプレットハンドル
 * @param[in]  isExtremity                       起動するのが末端の LA であれば true。そうでなければ false にする。
 */
nn::Result StartLibraryApplet(
    nn::applet::LibraryAppletHandle laHandle,
    bool isExtremity = false ) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief ライブラリアプレットをFG起動するための、簡易 API。サンプル実装です。
 *
 * @details  単に入力を渡して、出力を得るだけの場合の簡易アプレット起動 API です。
 *           多少特殊なことをしたい場合は、この API をLibraryAppletHandle を使って
 *           自前でハンドリングしてください。
 *
 * @param[in]  appletId                          ライブラリアプレットID
 * @param[in]  commonArg                         LA の共通パラメータ
 * @param[in]  laParam                           LA の入力パラメーター
 * @param[in]  laParamSize                       LA の出力パラメーターサイズ
 * @param[out] pResult                           結果を受け取るバッファ
 * @param[in]  resultBufferSize                  結果を格納するためのバッファサイズ
 * @param[out] pResultSize                       実際に受け取った結果のサイズ
 * @param[in]  isExtremity                       起動するのが末端の LA であれば true、そうでなければ false にする。
 */
nn::Result StartLibraryAppletEasy(
    nn::applet::AppletId appletId,
    const nn::la::CommonArgumentsWriter& commonArg,
    const void* laParam, size_t laParamSize,
    void* pResult, size_t resultBufferSize, size_t* pResultSize,
    bool isExtremity = false ) NN_NOEXCEPT;
//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief StartLibraryApplet の替りに JumpLibraryApplet を使うかどうかを設定します。
 *
 * @param[in] use JumpLibraryApplet を使うかどうかの設定
 */
void UseJumpLibraryApplet(bool use) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief SAにHOMEメニュー遷移をリクエストします。
 *        基本的にはハードのHOMEボタンをユーザーが押したのと同じ挙動になります。
 *        遷移するとは限りません。
 */
void RequestHomeMenu() NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @internal
 *
 * @brief SAに本体更新ジャンプをリクエストします。
 */
void RequestJumpToSystemUpdate() NN_NOEXCEPT;

}} // namespace nn::la

