﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/jit/jit_CommonTypes.h>
#include <nn/jit/jit_Result.h>

namespace nn { namespace jit {

/**
    @brief Jit 操作用ハンドルです。
*/
struct JitHandle
{
    void* _p;
};

/**
    @brief jit ライブラリを初期化します。

    @pre jit ライブラリが初期されていない
    @post jit ライブラリが初期化されている
*/
Result Initialize() NN_NOEXCEPT;

/**
    @brief jit ライブラリを終了します。

    @pre jit ライブラリが初期化されている
    @pre CreateJitEnvironment() に対応する DestroyJitEnvironment() が呼ばれている
    @post jit ライブラリが初期されていない
*/
void Finalize() NN_NOEXCEPT;

/**
    @brief Jit 環境のコンフィギュレーションを指定するための構造体です。
*/
struct JitEnvironmentConfiguration
{
    size_t rxCodeSize;
    size_t roCodeSize;
};

/**
    @brief Jit プラグインの情報を表す構造体です。
*/
struct JitPluginInfo
{
    void* nrr;
    size_t nrrSize;
    void* nro;
    size_t nroSize;
    size_t pluginWorkMemorySize;
};

/**
    @brief Jit 環境を作成します。

    @param[out] pOut JitHandle を受けるための変数へのポインタを指定します。
    @param[in] configuration Jit 環境のコンフィギュレーションを指定します。
    @param[in] pluginInfo プラグインの情報を指定します。
    @param[in] workMemory ワークメモリを指定します。
    @param[in] workMemorySize ワークメモリサイズを指定します。
    @param[in] option 動作オプションを指定します。

    @pre jit ライブラリが初期化されている

    @pre configuration.rxCodeSize が 2MiB でアラインされている
    @pre configuration.roCodeSize が 2MiB でアラインされている

    @pre [nrr, nrr + nrrSize) の領域に Jit Plugin DLL の nrr データが書き込まれている
    @pre [nro, nro + nroSize) の領域に Jit Plugin DLL の nro データが書き込まれている
    @pre pluginInfo.pluginWorkMemorySize が 2MiB でアラインされている

    @pre workMemory が 2MiB でアラインされている
    @pre workMemorySize が 2MiB でアラインされている
    @pre workMemorySize >= configuration.rxCodeSize + configuration.roCodeSize + pluginInfo.pluginWorkMemorySize
    @pre [workMemory, workMemory + workMemorySize) で指定される領域が読み書き可能である

    @post *pOut に Jit 環境を操作するための有効なハンドルが格納されている
    @post [workMemory, workMemory + workMemorySize) で指定される領域がライブラリの管理下に置かれる(通常アクセスはできなくなる)
*/
Result CreateJitEnvironment(JitHandle* pOut, const JitEnvironmentConfiguration& configuration, const JitPluginInfo& pluginInfo, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief Jit 環境を破棄します。

    @param[in] handle Jit ハンドルを指定します。

    @pre handle が有効なハンドルである
    @pre handle に対する操作が行われていない

    @post handle は無効なハンドルである
*/
void DestroyJitEnvironment(JitHandle handle) NN_NOEXCEPT;

/**
    @brief Jit 環境の情報を取得します。

    @param[in] handle Jit ハンドルを指定します。
    @return Jit 環境の情報を返します。

    @pre handle が有効なハンドルである

    @details
     同じ handle に対して本関数を複数回呼んだ場合、同一の値が返ることが保証されています。
*/
JitEnvironmentInfo GetJitEnvironmentInfo(JitHandle handle) NN_NOEXCEPT;

/**
    @brief Jit 環境を操作します。

    @pre handle が有効なハンドルである
*/
int Control(JitHandle handle, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT;

/**
    @brief Jit コード生成を行います。

    @param[out] pGeneratedRx 実際に生成された RX コード領域の範囲を格納する変数へのポインタを指定します。
    @param[out] pGeneratedRo 実際に生成された RO コード領域の範囲を格納する変数へのポインタを指定します。
    @param[in] handle Jit ハンドルを指定します。
    @param[in] tag タグを指定します。
    @param[in] source コード生成のソースデータを指定します。
    @param[in] sourceSize コード生成のソースデータのサイズを指定します。
    @param[in] rxBuffer RX コードを書き込む領域を指定します。
    @param[in] roBuffer RO コードを書き込む領域を指定します。
    @param[in] inData その他の入力データを指定します。
    @param[in] inDataSize その他の入力データのサイズを指定します。
    @param[in] outBuffer その他の出力を格納するバッファを指定します。
    @param[in] outBufferSize その他の出力を格納するバッファのサイズを指定します。

    @pre pGeneratedRx != nullptr
    @pre pGeneratedRo != nullptr
    @pre [source, source + sourceSize) の領域にコード生成のソースデータが格納されており、読み込みが可能である
    @pre handle が有効なハンドルである
    @pre rxBuffer.IsValid(GetJitEnvironmentInfo(handle).rxCodeSize)
    @pre roBuffer.IsValid(GetJitEnvironmentInfo(handle).roCodeSize)
    @pre inDataSize <= 32
    @pre [inData, inData + inDataSize) の領域にその他の入力データが格納されており、読み込みが可能である
    @pre [outBuffer, outBuffer + outBufferSize) の領域が書き込み可能である

    @post pGeneratedRx->IsInRange(rxBuffer)
    @post pGeneratedRo->IsInRange(roBuffer)

    @details
     本関数の呼び出し中に rxBuffer および roBuffer に対応するコード領域へのアクセスをしてはいけません。
*/
int GenerateCode(CodeRange* pGeneratedRx, CodeRange* pGeneratedRo, JitHandle handle, uint64_t tag, const void* source, size_t sourceSize, const CodeRange& rxBuffer, const CodeRange& roBuffer, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT;

}}
