﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/idle/idle_Result.h>
#include <nn/idle/idle_SystemTypes.h>

/**
 * @namespace nn::idle
 * @brief 無操作状態ハンドラの名前空間です。
 */

namespace nn { namespace idle {

    //! @name システムモジュール向けライブラリの初期化・終了
    //! @{

    /**
    * @brief システムモジュール向け機能ブロックを初期化します。
    *
    * @post
    *  対象の機能ブロックが初期化済の状態になります。
    *
    * @details
    *  システムモジュール向け機能ブロックを初期化します。 @n
    *  本関数の呼び出し回数はライブラリ内部でカウントされます。
    */
    void InitializeForSystem() NN_NOEXCEPT;

    /**
    * @brief システムモジュール向け機能ブロックを終了します。
    *
    * @post
    *  対象の機能ブロックが初期化前の状態になります。
    *
    * @details
    *  システムモジュール向け機能ブロックを終了します。 @n
    *  初期化を行った回数分、本関数が呼ばれた場合にのみ、対象の機能ブロックが初期化前の状態になります。
    */
    void FinalizeForSystem() NN_NOEXCEPT;

    //! @}

    //! @name SPSM 向け機能
    //! @{

    /**
    * @brief       指定したシステムイベントオブジェクトを、自動スリープ発動に紐づけて初期化します。
    *
    * @param[out]  pOutEvent 初期化対象のシステムイベント。
    * @param[in]   eventClearMode システムイベントの初期化時に指定するクリアモード
    *
    * @return
    *  処理の結果を返します。
    *
    * @pre
    *  - システムモジュール向け機能ブロックが初期化済
    *  - pOutEvent != nullptr
    *  - *pOutEvent が未初期化
    *
    * @post
    *  - *pOutEvent が初期化済
    *
    * @details
    *  指定したシステムイベントオブジェクトを、自動スリープ発動に紐づけて初期化します。
    *  自動スリープ発動時にシグナルされるようになります。
    */
    Result GetAutoPowerDownEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    //! @}

    //! @name AM/OMM 向け機能
    //! @{

    /**
    * @brief       ハンドリングコンテキストを更新します。
    *
    * @param[in]   handlingContext ハンドリングコンテキスト
    *
    * @return
    *  処理の結果を返します。
    *
    * @pre
    *  - システムモジュール向け機能ブロックが初期化済
    *  - handlingContext.overrideAutoSleepTimeInHandheldInSeconds >= 0
    *  - handlingContext.overrideAutoSleepTimeInConsoleInSeconds >= 0
    *  - handlingContext.overrideDimmingTimeInHandheldInSeconds >= 0
    *  - handlingContext.overrideDimmingTimeInConsoleInSeconds >= 0
    *  - handlingContext.overrideAutoSleepTimeInHandheldInSeconds < @ref nn::idle::MaxOverrideAutoSleepTimeInSeconds
    *  - handlingContext.overrideAutoSleepTimeInConsoleInSeconds < @ref nn::idle::MaxOverrideAutoSleepTimeInSeconds
    *  - handlingContext.overrideDimmingTimeInHandheldInSeconds < @ref nn::idle::MaxOverrideDimmingTimeInSeconds
    *  - handlingContext.overrideDimmingTimeInConsoleInSeconds < @ref nn::idle::MaxOverrideDimmingTimeInSeconds
    *
    * @details
    *  ハンドリングコンテキストを更新します。
    */
    Result SetHandlingContext(const HandlingContext& handlingContext) NN_NOEXCEPT;

    /**
    * @brief       現在の本体設定値を NAND から読み出し、現在の制御ポリシーに適用します。
    *
    * @return
    *  処理の結果を返します。
    *
    * @pre
    *  - システムモジュール向け機能ブロックが初期化済
    *
    * @details
    *  ユーザ設定値を NAND から読み出し、現在の制御ポリシーに適用します。
    */
    Result LoadAndApplySettings() NN_NOEXCEPT;

    /**
    * @brief       ユーザの操作があったことを明示的に宣言します。
    *
    * @return
    *  処理の結果を返します。
    *
    * @pre
    *  - システムモジュール向け機能ブロックが初期化済
    *
    * @details
    *  ユーザの操作があったことを明示的に宣言し、無操作時間のカウントをリセットします。
    */
    Result ReportUserIsActive() NN_NOEXCEPT;

    //! @}

}}
