﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief   GPIO パッドを表すオブジェクトクラスの宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ddsf/ddsf_IDevice.h>
#include <nn/gpio/driver/detail/gpio_PadSessionImpl.h>

//---------------------------------------------------------------------------

namespace nn {
namespace gpio {
namespace driver {

    class IGpioDriver;

    /**
    * @brief    ひとつの GPIO パッドを表すオブジェクトクラスです。
    *
    * @detail
    *   このオブジェクトはドライバ実装の提供側がリソースを確保し、ドライバ実装自身を表す IGpioDriver 型のドライバオブジェクトを紐づけておく必要があります。 @n
    *   @n
    *   本クラスを継承してドライバ実装側が独自に拡張した派生クラスを使用することもできます。 @n
    *   その場合、本クラスから派生クラスへのダウンキャストには、基底クラス @ref nn::ddsf::IDevice の提供する SafeCastTo<>(), SafeCastToPointer<>() メソッドを使用することを強く推奨します。 @n
    *   安全なダウンキャストを使用するには、派生クラスで @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用した型情報タグの定義が必要です。 @n
    *   詳細は @ref nn::ddsf::ICastSafe のリファレンスを参照してください。
    */
    class Pad :
        public nn::ddsf::IDevice
    {
        NN_DDSF_CAST_SAFE_DECL;

        NN_DISALLOW_COPY(Pad);
        NN_DISALLOW_MOVE(Pad);

    public:

        //! @name 生成と破棄
        //! @{

        /**
        * @brief    デフォルトコンストラクタです。
        */
        Pad() NN_NOEXCEPT :
            m_PadNumber(0),
            m_IsInterruptEnabled(false)
        {}

        /**
        * @brief    パラメータの設定を同時に行うコンストラクタです。
        *
        * @param[in]    padNumber   パッド番号
        *
        * @detail
        *   パラメータの設定を同時に行うコンストラクタです。
        */
        explicit Pad(int padNumber) NN_NOEXCEPT :
            m_PadNumber(padNumber),
            m_IsInterruptEnabled(false)
        {}

        /**
        * @brief    デストラクタです。
        */
        virtual ~Pad() NN_NOEXCEPT
        {}

        //! @}

        //! @name パラメータの設定・取得
        //! @{

        /**
        * @brief    GPIO パッド番号を取得します。
        *
        * @return   GPIO パッド番号
        *
        * @detail
        *   GPIO パッド番号を取得します。
        */
        int GetPadNumber() const NN_NOEXCEPT
        {
            return m_PadNumber;
        }

        /**
        * @brief    GPIO パッド番号を設定します。
        *
        * @param[in]    padNumber   パッド番号
        *
        * @detail
        *   GPIO パッド番号を設定します。 @n
        *   @n
        *   パッド番号は、ドライバ実装がパッド識別に使用できる任意の一意の整数を設定してください。 @n
        *   GPIO サブシステム側は、廃止予定 API である nn::gpio::driver::OpenSessionForDev() 呼び出し時にのみこの値を使用します。
        */
        void SetPadNumber(int padNumber) NN_NOEXCEPT
        {
            m_PadNumber = padNumber;
        }

        //! @}

        //! @name 割り込みの設定
        //! @{

        /**
        * @brief    GPIO パッドの割り込み許可状態を取得します。
        *
        * @return   GPIO パッドの割り込み許可状態
        *
        * @detail
        *   GPIO パッドの割り込み許可状態を取得します。 @n
        *   @n
        *   本関数が true を返しても、ドライバが実際に割り込みを有効化しているとは限りません。 @n
        *   ドライバに対して実際に割り込みを有効化させるかどうかの取得には IsInterruptRequiredEnabledOnDriver() を使用してください。 @n
        *   本関数は基本的に GPIO サブシステムのみが使用し、ドライバ実装では参照する必要はありません。
        */
        bool IsInterruptEnabled() const NN_NOEXCEPT
        {
            return m_IsInterruptEnabled;
        }

        /**
        * @brief    GPIO パッドの割り込み許可状態を設定します。
        *
        * @param[in]    GPIO パッドの割り込み許可状態
        *
        * @detail
        *   GPIO パッドの割り込み許可状態を設定します。 @n
        *   @n
        *   本関数に true をセットしても、ドライバがただちに割り込みを有効化するとは限りません。 @n
        *   ドライバに対して実際に割り込みを有効化させるかどうかの取得には IsInterruptRequiredEnabledOnDriver() を使用してください。 @n
        *   本関数は基本的に GPIO サブシステムのみが使用し、ドライバ実装では参照する必要はありません。
        */
        void SetInterruptEnabled(bool enable) NN_NOEXCEPT
        {
            m_IsInterruptEnabled = enable;
        }

        /**
        * @brief    ドライバに対して実際に GPIO パッドの割り込みを有効化させる必要があるかどうかを返します。
        *
        * @return   ドライバに対して実際に GPIO パッドの割り込みを有効化させる必要があるか
        *
        * @detail
        *   ドライバに対して実際に GPIO パッドの割り込みを有効化させる必要があるかどうかを返します。 @n
        *   本関数は基本的に GPIO サブシステムのみが使用し、ドライバ実装では参照する必要はありません。
        */
        bool IsInterruptRequiredEnabledOnDriver() const NN_NOEXCEPT
        {
            return IsInterruptEnabled() && IsAnySessionBoundToInterrupt();
        }

        /**
        * @brief    GPIO パッド上の割り込みに対してイベントをバインドしているセッションがいるかどうかを返します。
        *
        * @return   GPIO パッド上の割り込みに対してイベントをバインドしているセッションがいるか
        *
        * @pre
        *   - 対応するドライバ実装の IGpioDriver::GetInterruptControlMutex() が返すミューテックスのロックを取得済
        *
        * @detail
        *   GPIO パッド上の割り込みに対してイベントをバインドしているセッションがいるかどうかを返します。 @n
        *   本関数を呼び出す際は、 IGpioDriver::GetInterruptControlMutex() が返すミューテックスのロックを取得してください。 @n
        *   本関数は基本的に GPIO サブシステムのみが使用し、ドライバ実装では参照する必要はありません。
        */
        bool IsAnySessionBoundToInterrupt() const NN_NOEXCEPT;

        /**
        * @brief    GPIO パッド上の割り込みに対してバインドしているセッションのイベントをシグナルします。
        *
        * @pre
        *   - 対応するドライバ実装の IGpioDriver::GetInterruptControlMutex() が返すミューテックスのロックを取得済
        *
        * @detail
        *   GPIO パッド上の割り込みに対してバインドしているセッションのイベントをシグナルします。 @n
        *   ドライバ実装は、割り込みハンドラの中で、割り込みをクリアした際にこちらを呼び出し、バインドしているセッションに通知してください。 @n
        *   本関数を呼び出す際は、 IGpioDriver::GetInterruptControlMutex() が返すミューテックスのロックを取得してください。
        */
        void SignalInterruptBoundEvent() NN_NOEXCEPT;

        //! @}

    private:
        int m_PadNumber; //!< パッド番号（番号と実体の対応はドライバ定義）
        bool m_IsInterruptEnabled{ false }; //!< 割り込みの有効化を要求されているか
    };

} // driver
} // gpio
} // nn
