﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバ実装提供者向け API に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/ddsf/ddsf_IEventHandler.h>
#include <nn/gpio/driver/gpio_IGpioDriver.h>
#include <nn/gpio/driver/detail/dt/gpio_DeviceCodeNodeParser.h>

namespace nn {
namespace gpio {
namespace driver {

//! @name GPIO ドライバ実装提供者向け API
//! @{

/**
* @brief GPIO サブシステムにドライバを登録します。
*
* @param[in]    pDriver ドライバオブジェクト
*
* @pre
*   - pDriver != nullptr
*   - nn::gpio::driver::Initialize() の実行前
*
* @detail
*   GPIO サブシステムにドライバを登録します。 @n
*   ドライバの登録は、GPIO サブシステムの初期化前（ @ref nn::gpio::driver::Initialize() が呼ばれる前）に行ってください。 @n
*   nn::gpio::driver::Initialize() の内部で、登録された各ドライバの初期化処理が呼ばれるためです。
*/
void RegisterDriver(IGpioDriver* pDriver) NN_NOEXCEPT;

/**
* @brief GPIO サブシステムへのドライバ登録を解除します。
*
* @param[in]    pDriver ドライバオブジェクト
*
* @pre
*   - pDriver != nullptr
*   - nn::gpio::driver::Initialize() の実行前
*   - pDriver が RegisterDriver() で登録済
*
* @detail
*   GPIO サブシステムへのドライバ登録を解除します。 @n
*   nn::gpio::driver::Initialize() の完了後で、サブシステムが稼働中は、ドライバの登録解除はできません。
*/
void UnregisterDriver(IGpioDriver* pDriver) NN_NOEXCEPT;

/**
* @brief GPIO サブシステムにデバイスコードノードパーサを登録します。
*
* @param[in]    pParser デバイスコードノードパーサ
*
* @pre
*   - pParser != nullptr
*   - 以下のいずれかの状態下での呼び出しである
*       - nn::gpio::driver::Initialize() の実行前
*       - nn::gpio::driver::Initialize() 内で呼び出される、各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中からの呼び出し
*
* @detail
*   GPIO サブシステムにデバイスコードノードパーサを登録します。 @n
*   デバイスコードノードパーサの登録は、GPIO サブシステムの初期化前（ @ref nn::gpio::driver::Initialize() が呼ばれる前）に行うか、 @n
*   サブシステム初期化時の各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中で行ってください。 @n
*   nn::gpio::driver::Initialize() の内部で、各ドライバの初期化処理を実行した直後にデバイスコードノードのパース処理を実行するためです。
*/
void RegisterDeviceCodeNodeParser(nn::gpio::driver::detail::dt::IDeviceCodeNodeParser* pParser) NN_NOEXCEPT;

/**
* @brief GPIO サブシステムにデバイスコードエントリを登録します。
*
* @param[in]    deviceCode  デバイスコード
* @param[in]    pPad        対応するパッドオブジェクト
*
* @retresult
*   @handleresult{nn::ResultSuccess}
*   @handleresult{nn::ddsf::ResultOutOfResource, デバイスコードエントリの空きがありません。}
* @endretresult
*
* @pre
*   - pPad != nullptr
*   - pPad->IsDriverAttached() == true
*   - 以下のいずれかの状態下での呼び出しである
*       - nn::gpio::driver::Initialize() 内で呼び出される、各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中からの呼び出し
*       - nn::gpio::driver::Initialize() 内で呼び出される、デバイスコードノードパーサによるパース処理中からの呼び出しである
*
* @detail
*   GPIO サブシステムにデバイスコードエントリを登録します。 @n
*   これによりクライアントは、 nn::gpio::OpenSession() にデバイスコードを渡すことで、関連づけたパッドオブジェクトに対してセッションを開くことができるようになります。 @n
*   サブシステム初期化時の各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中で行ってください。 @n
*   nn::gpio::driver::Initialize() の内部で、各ドライバの初期化処理を実行した直後にデバイスコードノードのパース処理を実行するためです。
*/
nn::Result RegisterDeviceCode(nn::DeviceCode deviceCode, Pad* pPad) NN_NOEXCEPT;

/**
* @brief    GPIO サブシステムから指定したデバイスコードのエントリを削除します。
*
* @param[in]    deviceCode  デバイスコード
*
* @return   該当するデバイスコードが存在し、削除が行われた場合に true を返します。該当デバイスコードがすでに存在しなかった場合 false を返します。
*
* @pre
*   - 以下のいずれかの状態下での呼び出しである
*       - nn::gpio::driver::Initialize() 内で呼び出される、各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中からの呼び出し
*       - nn::gpio::driver::Initialize() 内で呼び出される、デバイスコードノードパーサによるパース処理中からの呼び出しである
*
* @detail
*   GPIO サブシステムから指定したデバイスコードのエントリを削除します。
*/
bool UnregisterDeviceCode(nn::DeviceCode deviceCode) NN_NOEXCEPT;

/**
* @brief GPIO サブシステムの割り込みハンドリング用スレッドにイベントハンドラを登録します。
*
* @param[in]    pHandler    イベントハンドラオブジェクト
*
* @pre
*   - pHandler != nullptr
*   - pHandler の nn::ddsf::IEventHandler::IsInitialized() == true
*   - pHandler の nn::ddsf::IEventHandler::IsRegistered() == false
*   - 以下のいずれかの状態下での呼び出しである
*       - nn::gpio::driver::Initialize() 内で呼び出される、各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中からの呼び出し
*       - nn::gpio::driver::Initialize() の完了後
*
* @post
*   - pHandler の nn::ddsf::IEventHandler::IsRegistered() == true
*
* @detail
*   GPIO サブシステムの割り込みハンドリング用スレッドにイベントハンドラを登録します。 @n
*   このメソッドを実行すると @a pHandler 内の多重待ちオブジェクトはリンク済の状態になります。 @n
*   @n
*   割り込みハンドリング用スレッドは登録された多重待ちオブジェクトに対してウェイトし、シグナル時に pHandler の nn::ddsf::IEventHandler::HandleEvent() を呼び出します。 @n
*   割り込みハンドリング用スレッドは通常非常に高優先度であるため、 pHandler の nn::ddsf::IEventHandler::HandleEvent() は時間のかかる処理を行ってはいけません。 @n
*   @n
*   割り込みハンドリング用スレッドは nn::gpio::driver::Initialize() の最後に開始されます。 @n
*   サブシステムの初期化完了後に本メソッドを呼び出した場合は、割り込みハンドリング用スレッド側と同期して、多重待ちを一時的に停止した状態で登録処理を行います。 @n
*   この関数を割り込みハンドリング用スレッド上で呼び出した場合、すなわち nn::ddsf::IEventHandler::HandleEvent() 内で呼び出した場合は、 @n
*   呼び出したコンテキスト上で登録処理を行います。
*/
void RegisterInterruptHandler(nn::ddsf::IEventHandler* pHandler) NN_NOEXCEPT;

/**
* @brief GPIO サブシステムの割り込みハンドリング用スレッドからイベントハンドラの登録を解除します。
*
* @param[in]    handler     イベントハンドラオブジェクト
*
* @pre
*   - pHandler != nullptr
*   - pHandler の nn::ddsf::IEventHandler::IsInitialized() == true
*   - pHandler の nn::ddsf::IEventHandler::IsRegistered() == true
*   - 以下のいずれかの状態下での呼び出しである
*       - nn::gpio::driver::Initialize() 内で呼び出される、各登録済ドライバの初期化処理 IGpioDriver::InitializeDriver() 中からの呼び出し
*       - nn::gpio::driver::Initialize() の完了後
*
* @post
*   - pHandler の nn::ddsf::IEventHandler::IsRegistered() == false
*
* @detail
*   GPIO サブシステムの割り込みハンドリング用スレッドからイベントハンドラの登録を解除します。 @n
*   このメソッドを実行すると @a pHandler 内の多重待ちオブジェクトは未リンクの状態になります。 @n
*   @n
*   割り込みハンドリング用スレッドは nn::gpio::driver::Initialize() の最後に開始されます。 @n
*   サブシステムの初期化完了後に本メソッドを呼び出した場合は、割り込みハンドリング用スレッド側と同期して、多重待ちを一時的に停止した状態で解除処理を行います。 @n
*   この関数を割り込みハンドリング用スレッド上で呼び出した場合、すなわち nn::ddsf::IEventHandler::HandleEvent() 内で呼び出した場合は、 @n
*   呼び出したコンテキスト上で解除処理を行います。
*/
void UnregisterInterruptHandler(nn::ddsf::IEventHandler* pHandler) NN_NOEXCEPT;

//! @}

} // driver
} // gpio
} // nn
