﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/ddsf/ddsf_ISession.h>
#include <nn/gpio/driver/gpio_PadAccessor.h> // for GpioPadSessionSize
#include <nn/gpio/driver/detail/gpio_EventHolder.h>

//---------------------------------------------------------------------------

namespace nn {
namespace gpio {
namespace driver {

    class Pad;

namespace detail {

    class PadSessionImpl final :
        public nn::ddsf::ISession
    {
        NN_DDSF_CAST_SAFE_DECL;

        NN_DISALLOW_COPY(PadSessionImpl);
        NN_DISALLOW_MOVE(PadSessionImpl);

    public:
        PadSessionImpl() NN_NOEXCEPT;
        ~PadSessionImpl() NN_NOEXCEPT;

        nn::Result Open(Pad* pPad, nn::ddsf::AccessMode accessMode) NN_NOEXCEPT;
        void Close() NN_NOEXCEPT;

        nn::Result BindInterrupt(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;
        void UnbindInterrupt() NN_NOEXCEPT;
        bool IsInterruptBound() const NN_NOEXCEPT
        {
            return m_EventHolder.IsBoundEvent();
        }
        nn::Result GetInterruptEnabled(bool* pOut) const NN_NOEXCEPT;
        nn::Result SetInterruptEnabled(bool enable) NN_NOEXCEPT;
        void SignalInterruptBoundEvent() NN_NOEXCEPT;

    private:
        nn::Result UpdateInterruptEnabledOnDriver() NN_NOEXCEPT;

    private:
        EventHolder m_EventHolder;
    };

    struct NN_ALIGNAS(GpioPadSessionAlign) GpioPadSessionImplPadded
    {
        PadSessionImpl impl;

        // 64bit ビルド時にサイズがかわるため、パディングを入れる。
        char           _padding[GpioPadSessionSize - sizeof(PadSessionImpl)];
    };

    inline PadSessionImpl& GetPadSessionImpl(GpioPadSession& session) NN_NOEXCEPT
    {
        return nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;
    }

    inline PadSessionImpl& GetPadSessionImplWithOpenCheck(GpioPadSession& session) NN_NOEXCEPT
    {
        auto& impl = GetPadSessionImpl(session);
        NN_SDK_ASSERT(impl.IsOpen());
        return impl;
    }

    inline const PadSessionImpl& GetPadSessionImpl(const GpioPadSession& session) NN_NOEXCEPT
    {
        return nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;
    }

    inline const PadSessionImpl& GetPadSessionImplWithOpenCheck(const GpioPadSession& session) NN_NOEXCEPT
    {
        const auto& impl = GetPadSessionImpl(session);
        NN_SDK_ASSERT(impl.IsOpen());
        return impl;
    }

    //---------------------------------------------------------------------------
    // static_assert で TypedStorage のサイズとアラインを確認
    //---------------------------------------------------------------------------

    NN_STATIC_ASSERT(sizeof(GpioPadSessionImplPadded) == GpioPadSessionSize);
    NN_STATIC_ASSERT(sizeof(PadSessionImpl) <= GpioPadSessionSize);
    NN_STATIC_ASSERT(NN_ALIGNOF(GpioPadSessionImplPadded) == GpioPadSessionAlign);
    NN_STATIC_ASSERT(NN_ALIGNOF(PadSessionImpl) <= GpioPadSessionAlign);

} // detail
} // driver
} // gpio
} // nn
