﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  システムセーブデータ関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace fs {

//! @name システムセーブデータ関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      ユーザーアカウントごとのシステムセーブデータをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  id         システムセーブデータ ID
*   @param[in]  user       ユーザーアカウント ID
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*       - user が指すユーザーアカウントが存在する。
*       - 対象のシステムセーブデータが存在する。
*/
Result MountSystemSaveData(const char* name, SystemSaveDataId id, const nn::account::Uid& user) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ユーザーアカウントごとのシステムセーブデータを作成します。
*
*   @param[in]  id           システムセーブデータ ID
*   @param[in]  user         ユーザーアカウント ID
*   @param[in]  size         セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize  セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags        セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*       - user が指すユーザーアカウントが存在する。
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SystemSaveDataId id, const nn::account::Uid& user, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ユーザーアカウントごとのシステムセーブデータを削除します。
*
*   @param[in]  id           システムセーブデータ ID
*   @param[in]  user         ユーザーアカウント ID
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*       - user != nn::account::InvalidUid
*       - 対象のシステムセーブデータが存在する。
*/
Result DeleteSystemSaveData(SystemSaveDataId id, const nn::account::Uid& user) NN_NOEXCEPT;

//! @}


}} // namespace nn::fs
