﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/fssrv/sf/fssrv_ISaveDataTransfer.h>
#include <nn/fs/fs_SaveDataTransferType.h>

namespace nn { namespace fssrv { namespace sf {
    class ISaveDataTransferManager;
    class ISaveDataExporter;
    class ISaveDataImporter;
}}}

namespace nn { namespace fs {

//! @name セーブデータ転送関連 API
//! @{

    using SaveDataChunkId = uint16_t;

    const SaveDataChunkId SaveDataChunkIdForInitialData = 0;

    // TORIAEZU: 固定長構造体
    struct InitialDataVersion2
    {
    public:
        static const int Size = 8192;
    private:
        char data[Size] NN_IS_UNUSED_MEMBER;
    };
    NN_STATIC_ASSERT(std::is_pod<InitialDataVersion2>::value);
    NN_STATIC_ASSERT(sizeof(InitialDataVersion2) == InitialDataVersion2::Size);

    class ISaveDataChunkIterator
    {
    public:
        virtual ~ISaveDataChunkIterator() NN_NOEXCEPT {}
        virtual const SaveDataChunkId GetId() const NN_NOEXCEPT = 0;
        virtual void Next() NN_NOEXCEPT = 0;
        virtual bool IsEnd() const NN_NOEXCEPT = 0;
    };

    class ISaveDataChunkExporter
    {
    public:
        virtual ~ISaveDataChunkExporter() NN_NOEXCEPT {}

        /**
        * @brief        チャンクデータの読み込みを行います。
        *
        * @param[out]   outValue        読み込んだサイズ
        * @param[in]    buffer          読み込みバッファ
        * @param[in]    size            読み込みバッファのサイズ
        *
        * @details      データの読み込みは複数回に分けて読み込むことが可能です。@n
        *               outValue に 0 が返るまで本関数を繰り返し呼び出してください。@n
        */
        virtual Result Pull(size_t* outValue, void* buffer, size_t size) NN_NOEXCEPT = 0;

        /*
        * @brief チャンクの生データの残バイトサイズを取得します。
        * @details Pull() の残量とは一致しません。進捗表示などにのみ利用してください。
        */
        virtual int64_t GetRestRawDataSize() NN_NOEXCEPT = 0;
    };

    class ISaveDataChunkImporter
    {
    public:
        virtual ~ISaveDataChunkImporter() NN_NOEXCEPT {}

        /**
        * @brief        チャンクデータの書き込みを行います。
        *
        * @param[in]    buffer          チャンクデータ
        * @param[in]    size            チャンクデータのサイズ
        *
        * @details      データの書き込みは複数回に分けて書き込むことが可能です。@n
        *               途中で書き込みに失敗したり、Prohibiter によって中断したりした場合、
        *               続きからデータの書き込みを継続することはできません。@n
        *               再度書き込みを行う場合は新たに ChunkImporter を開き直して最初からやり直してください。@n
        */
        virtual Result Push(const void* buffer, size_t size) NN_NOEXCEPT = 0;

    };

    class ISaveDataDivisionExporter
    {
    public:
        using KeySeed = nn::fs::detail::KeySeed;
        using InitialDataMac = nn::fs::detail::InitialDataMac;
        using InitialDataAad = nn::fs::detail::InitialDataAad;

    public:
        virtual ~ISaveDataDivisionExporter() NN_NOEXCEPT {}

        /*
        * @brief エクスポートするセーブデータの分割数を設定します。
        *
        * @pre
        *       - エクスポータが無効でない
        */
        virtual void SetDivisionCount(int count) NN_NOEXCEPT = 0;

        /**
        * @brief 差分のあるチャンクを列挙するイテレータを取得します。
        *
        * @pre
        *       - エクスポータが無効でない
        */
        virtual Result OpenSaveDataDiffChunkIterator(std::unique_ptr<ISaveDataChunkIterator>* outValue) NN_NOEXCEPT = 0;

        /**
        * @brief 指定 SaveDataChunkId のチャンクに対するエクスポータを取得します。
        *
        * @pre
        *       - エクスポータが無効でない
        */
        virtual Result OpenSaveDataChunkExporter(std::unique_ptr<ISaveDataChunkExporter>* outValue, SaveDataChunkId id) NN_NOEXCEPT = 0;

        /**
        * @brief 新規エクスポート処理を完了し、keysSeed と initialDataMac を取得します。
        *
        * @pre
        *       - 新規エクスポートとして開いたエクスポータである
        *       - OpenSaveDataDiffChunkIterator() が返す全てのチャンクについて Pull() し終えている
        *       - エクスポータが無効でない
        *
        * @post
        *       - ResultSuccess が返った場合、エクスポータが無効である
        */
        virtual Result FinalizeFullExport(KeySeed* pOutKeySeed, InitialDataMac* pOutInitialDataMac) NN_NOEXCEPT = 0;

        /**
        * @brief 差分エクスポート処理を完了し、 initialDataMac を取得します。
        *
        * @pre
        *       - 差分エクスポートとして開いたエクスポータである
        *       - OpenSaveDataDiffChunkIterator() が返す全てのチャンクについて Pull() し終えている
        *       - エクスポータが無効でない
        *
        * @post
        *       - ResultSuccess が返った場合、エクスポータが無効である
        */
        virtual Result FinalizeDiffExport(InitialDataMac* pOutInitialDataMac) NN_NOEXCEPT = 0;

        /**
        * @brief エクスポート処理をキャンセルします。
        *
        * @pre
        *       - エクスポータが無効でない
        *
        * @post
        *       - エクスポータが無効である
        */
        virtual Result CancelExport() NN_NOEXCEPT = 0;

        /**
        * @brief エクスポータのオープン時に渡した InitialData 内の追加認証データを取得します。
        *
        * @pre
        *       - 差分エクスポートとして開いたエクスポータである
        *       - エクスポータが無効でない
        */
        virtual Result GetImportInitialDataAad(InitialDataAad* pOutValue) NN_NOEXCEPT = 0;

        /**
        * @brief    エクスポートされる InitialData に埋め込む追加認証データを設定します。
        *
        * @details  設定した追加認証データは id が nn::fs::SaveDataChunkIdForInitialData のチャンクに含まれるため、
        *           nn::fs::SaveDataChunkIdForInitialData のチャンクについて Pull() を行う前に設定してください。
        *
        * @pre
        *           - エクスポータが無効でない
        */
        virtual Result SetExportInitialDataAad(const InitialDataAad& aad) NN_NOEXCEPT = 0;
    };

    class ISaveDataDivisionImporter
    {
    public:
        using InitialDataAad = nn::fs::detail::InitialDataAad;

    public:
        virtual ~ISaveDataDivisionImporter() NN_NOEXCEPT {}

        /**
        * @brief        差分のあるチャンクを列挙するイテレータを取得します。
        *
        * @pre
        *               - インポータが無効でない
        */
        virtual Result OpenSaveDataDiffChunkIterator(std::unique_ptr<ISaveDataChunkIterator>* outValue) NN_NOEXCEPT = 0;

        /**
        *   @brief      インポートのための初期化を行います。
        *
        *   @details    対象セーブデータのインポートを開始するための初期化処理を行います。
        *               この関数を実行するまで、対象の作成・更新処理は発生しません。@n
        *               初期化完了に必要な処理量のうち、 processSize バイトを 1 回の呼び出しで処理します。
        *               1 回の呼び出しにかかる時間は processSize に応じて変わります。
        *               outRestSize に 0 が返るまで本関数を繰り返し呼び出してください。@n
        *
        *   @param[out] outRestSize         初期化処理の残りのバイトサイズ
        *   @param[in]  processSize         処理のバイトサイズ
        *
        *   @return     処理の結果が返ります。
        *   @retval     ResultSuccess               成功しました。
        *   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
        *
        *   @pre
        *               - インポート未初期化状態である
        *               - インポータが無効でない
        *
        *   @post
        *               - outRestSize に 0 が返った場合、インポート初期化完了状態である
        */
        virtual Result InitializeImport(int64_t* outRestSize, int64_t processSize) NN_NOEXCEPT = 0;

        // obsolete
        virtual Result InitializeImport(int64_t* outRequiredSize) NN_NOEXCEPT = 0;

        /**
        * @brief        インポート処理を確定します。
        *
        * @pre
        *               - インポート初期化完了状態である
        *               - インポータが無効でない
        *
        * @post
        *               - ResultSuccess が返った場合、インポータが無効である
        */
        virtual Result FinalizeImport() NN_NOEXCEPT = 0;

        /**
        * @brief        インポート処理をキャンセルします。
        *
        * @details      書き込み途中のデータを破棄し、インポート処理開始前の状態に戻します。
        *
        * @pre
        *               - InitializeImport() が実行済み
        *               - インポータが無効でない
        *
        * @post
        *               - インポータが無効である
        */
        virtual Result CancelImport() NN_NOEXCEPT = 0;

        /**
        * @brief        指定 SaveDataChunkId のチャンクに対するインポータを取得します。
        *
        * @pre
        *               - インポート初期化完了状態である
        *               - インポータが無効でない
        */
        virtual Result OpenSaveDataChunkImporter(std::unique_ptr<ISaveDataChunkImporter>* outValue, SaveDataChunkId id) NN_NOEXCEPT = 0;

        /**
        * @brief        インポータのオープン時に渡した InitialData 内の追加認証データを取得します。
        *
        * @pre
        *               - インポータが無効でない
        */
        virtual Result GetImportInitialDataAad(InitialDataAad* pOutValue) NN_NOEXCEPT = 0;
    };


    class SaveDataTransferManagerVersion2 : public detail::Newable
    {
    public:

        struct Challenge
        {
        public:
            static const int Size = 16;

        public:
            char data[Size] NN_IS_UNUSED_MEMBER;
        };

        struct KeySeedPackage
        {
        public:
            static const int Size = 512;

        public:
            char data[Size] NN_IS_UNUSED_MEMBER;
        };

        // obsolete
        using Token = KeySeedPackage;

    public:
        SaveDataTransferManagerVersion2() NN_NOEXCEPT;

        /*
        * @brief サーバに KeySeedPackage をリクエストする際に利用するチャレンジを取得します。
        */
        Result GetChallenge(Challenge* outValue) NN_NOEXCEPT;

        /*
        * @brief GetChallenge() で取得したチャレンジに対応する KeySeedPackage をセットします。
        */
        Result SetKeySeedPackage(const KeySeedPackage& pToken) NN_NOEXCEPT;

        // obsolete
        Result SetToken(const Token& pToken) NN_NOEXCEPT
        {
            return SetKeySeedPackage(pToken);
        }

        /*
        * @brief 新規エクスポートのためのエクスポータを取得します。
        * @pre
        */
        Result OpenSaveDataFullExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

        /*
        * @brief 差分更新エクスポートのためのエクスポータを取得します。
        * @pre
        *               - SetKeySeedPackage() が実行済み
        */
        Result OpenSaveDataDiffExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;


        /*
        * @brief 新規インポートのためのインポータを取得します。
        * @pre
        *               - SetKeySeedPackage() が実行済み
        */
        Result OpenSaveDataFullImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, const account::Uid& uid, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

        /*
        * @brief 差分更新インポートのためのインポータを取得します。
        * @pre
        *               - SetKeySeedPackage() が実行済み
        */
        Result OpenSaveDataDiffImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

        /*
        * @brief 差分更新インポートのためのインポータを取得します。
        * @pre
        *               - SetKeySeedPackage() が実行済み
        */
        Result OpenSaveDataDuplicateDiffImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;


    public:
        // obsolete
        inline Result OpenSaveDataExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
        {
            return OpenSaveDataFullExporter(outValue, saveDataSpaceId, saveDataId);
        }
        inline Result OpenSaveDataExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
        {
            return OpenSaveDataDiffExporter(outValue, initialData, saveDataSpaceId, saveDataId);
        }

        inline Result OpenSaveDataImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, const account::Uid& uid, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
        {
            return OpenSaveDataFullImporter(outValue, initialData, uid, saveDataSpaceId);
        }
        inline Result OpenSaveDataImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
        {
            return OpenSaveDataDiffImporter(outValue, initialData, saveDataSpaceId, saveDataId);
        }

    public:
        Result OpenSaveDataFullImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, const fs::UserId& userId, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataTransferManagerWithDivision> m_Interface;

    };

    //--------------------------------------------------------------------------
    /**
    * @brief 生存する間、対象アプリケーションのセーブデータのインポート・エクスポートを禁止します。破棄することで解除されます。
    */
    class SaveDataTransferProhibiterForCloudBackUp
    {
    public:
        explicit SaveDataTransferProhibiterForCloudBackUp(sf::SharedPointer<fssrv::sf::ISaveDataTransferProhibiter> pInterface)
            : m_Interface(std::move(pInterface))
        {
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataTransferProhibiter> m_Interface;
    };

    //--------------------------------------------------------------------------
    /**
    *   @brief      指定したアプリケーションのセーブデータのインポート・エクスポートを禁止する SaveDataTransferProhibiterForCloudBackUp をオープンします。
    *
    *   @param[out] outValue        prohibiter
    *   @param[in]  applicationId   対象のアプリケーション ID
    *
    *   @return     処理の結果が返ります。
    *   @retval     ResultSuccess               成功しました。
    *
    */
    Result OpenSaveDataTransferProhibiterForCloudBackUp(std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp>* outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      指定したアプリケーションのセーブデータのインポート・エクスポートを禁止する SaveDataTransferProhibiterForCloudBackUp をオープンします。
    *
    *   @param[out] outValue        prohibiter の配列
    *   @param[in]  applicationIds  対象のアプリケーション ID
    *
    *   @return     処理の結果が返ります。
    *   @retval     ResultSuccess               成功しました。
    *
    */
    Result OpenSaveDataTransferProhibiterForCloudBackUp(std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> outValue[], nn::ncm::ApplicationId applicationIdList[], int count) NN_NOEXCEPT;

    // obsolete
    typedef SaveDataTransferProhibiterForCloudBackUp SaveDataExportProhibiter;
    Result OpenSaveDataExportProhibiter(std::unique_ptr<SaveDataExportProhibiter>* outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    Result OpenSaveDataExportProhibiter(std::unique_ptr<SaveDataExportProhibiter> outValue[], nn::ncm::ApplicationId applicationIdList[], int count) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
     *   @brief      指定したアプリケーションのプログラムが参照可能なセーブデータの所有者 ID を取得します。
     *
     *   @param[out] outValue        outList に格納されたセーブデータ所有者 ID の個数
     *   @param[out] outList         取得したセーブデータ所有者 ID を格納する配列
     *   @param[in]  applicationId   対象のアプリケーション ID
     *   @param[in]  programIndex    プログラムインデックス
     *   @param[in]  offset          取得を省略するセーブデータ所有者 ID の個数
     *   @param[in]  count           取得するセーブデータ所有者 ID の個数
     *
     *   @return     処理の結果が返ります。
     *   @retval     ResultSuccess               成功しました。
     *
     *   @pre 対象のプログラムが動作するプロセスが起動されている
     *   @pre outList != nullptr && outList のサイズが sizeof(nn::ncm::ApplicationId) * count 以上である
     *   @pre offset >= 0
     */
    Result ListApplicationAccessibleSaveDataOwnerId(int* outValue, nn::ncm::ApplicationId outList[], nn::ncm::ApplicationId applicationId, int programIndex, int offset, int count) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
     *   @brief      指定したアプリケーションのプログラムが参照可能なセーブデータの所有者 ID の個数を取得します。
     *
     *   @param[out] outValue        セーブデータ所有者 ID の個数
     *   @param[in]  applicationId   対象のアプリケーション ID
     *   @param[in]  programIndex    プログラムインデックス
     *
     *   @return     処理の結果が返ります。
     *   @retval     ResultSuccess               成功しました。
     *
     *   @pre 対象のプログラムが動作するプロセスが起動されている
     */
    Result GetCountOfApplicationAccessibleSaveDataOwnerId(int* outValue, nn::ncm::ApplicationId applicationId, int programIndex) NN_NOEXCEPT;

//! @}

}}
