﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fssrv { namespace sf {
    class ISaveDataTransferManager;
    class ISaveDataExporter;
    class ISaveDataImporter;
}}}

namespace nn { namespace fs {

//! @name セーブデータ転送関連 API
//! @{
    struct InitialData
    {
    public:
        static const int Size = 640;

    public:
        struct {
            SaveDataExtraData extraData;
            uint32_t version;
            char reserved[108];
        } encryptedArea;
        char mac[16];
    };
    NN_STATIC_ASSERT(std::is_pod<InitialData>::value);
    NN_STATIC_ASSERT(sizeof(InitialData) == InitialData::Size);


    /**
     * @brief セーブデータの転送に必要なサイズを取得するためのユーティリティクラスです。
     */
    class SaveDataTransferSizeCalculator
    {
    public:
        /*
         * @brief 指定したセーブデータ情報が示すセーブデータをエクスポートするのに必要な転送量を取得します。
         *
         * @pre
         *     - info が現在存在するセーブデータの SaveDataInfo である
         *     - 当該セーブデータがマウントされていない
         *     - 当該セーブデータが SaveDataImporter/SaveDataExporter によって開かれていない
         *
         * @return 取得した値が返ります。
         */
        static int64_t QuerySaveDataExportSize(const SaveDataInfo& info) NN_NOEXCEPT;

        /*
         * @brief 指定したセーブデータ情報が示すセーブデータをインポートするのに必要な容量を取得します。
         *
         * @return 取得した値が返ります。
         */
        static int64_t QuerySaveDataRequiredSizeForImport(const SaveDataInfo& info) NN_NOEXCEPT;

        /*
         * @brief 指定したセーブデータ情報が示すセーブデータをエクスポートするのに必要な転送量を取得します。
         *
         * @pre
         *     - info が現在存在するセーブデータの SaveDataInfo である
         *     - 当該セーブデータがマウントされていない
         *     - 当該セーブデータが SaveDataImporter/SaveDataExporter によって開かれていない
         *
         * @return 取得した値が返ります。
         */
        static int64_t QuerySaveDataExportSize(const SaveDataInfo* info, uint32_t count) NN_NOEXCEPT;

        /*
         * @brief 指定したセーブデータ情報が示すセーブデータをインポートするのに必要な容量を取得します。
         *
         * @return 取得した値が返ります。
         */
        static int64_t QuerySaveDataRequiredSizeForImport(const SaveDataInfo* info, uint32_t count) NN_NOEXCEPT;

        /*
         * @brief セーブデータ領域の空き領域を取得します。
         *
         * @return 取得した値が返ります。
         *
         * @details 取得した値が QuerySaveDataRequiredSizeForImport で取得した値以上であれば、その時点においてインポートが可能です。
         */
        static int64_t GetFreeSpaceSize(SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;
    };

    class SaveDataTransferManager;

    /**
     * @brief セーブデータのエクスポーターです。
     *
     * @details このクラスのインスタンスは SaveDataTransferManager::OpenSaveDataExporter で取得します。@n
     *          Pull, PullInitialData で出力されるバイト列は SaveDataTransferManager に設定されたトークンに基づいて署名・暗号化されます。@n
     */
    class SaveDataExporter : public detail::Newable
    {
    public:
        friend SaveDataTransferManager;
        static const size_t InitialDataSize = InitialData::Size;

    public:
        /*
         * @brief エクスポート対象のセーブデータ情報を取得します。
         *
         * @return 取得した情報の参照が返ります。
         */
        const SaveDataInfo& GetSaveDataInfo() NN_NOEXCEPT;

        /**
         * @brief インポーター向けの初期化データを取得します。
         *
         * @pre size >= InitialDataSize
         *
         * @return 処理の結果が返ります。
         */
        Result PullInitialData(void* buffer, size_t size) NN_NOEXCEPT;

        /**
         * @brief エクスポートデータストリームからデータを読み出し進めます。
         *
         * @return 処理の結果が返ります。
         *
         * @details 処理が成功した場合 outPulledSize に読み出せたバイトサイズが格納されます。@n
         *          処理が失敗した場合 outPulledSize の値は不定です。
         */
        Result Pull(size_t* outPulledSize, void* buffer, size_t size) NN_NOEXCEPT;

        /*
         * @brief エクスポートデータストリームの残りのバイトサイズを取得します。
         *
         * @return 取得した値が返ります。
         */
        int64_t GetRestSize() NN_NOEXCEPT;

    private:
        NN_IMPLICIT SaveDataExporter(sf::SharedPointer<fssrv::sf::ISaveDataExporter>&& interface) NN_NOEXCEPT;
        sf::SharedPointer<fssrv::sf::ISaveDataExporter> m_Interface;
        SaveDataInfo m_SaveDataInfo;
    };

    /**
     * @brief セーブデータのインポーターです。
     *
     * @details このクラスのインスタンスは SaveDataTransferManager::OpenSaveDataImporter で取得します。@n
     *          Push で入力されるバイト列は SaveDataTransferManager に設定されたトークンに基づいて復号化され、Finalize 実行時に検証されます。
     */
    class SaveDataImporter : public detail::Newable
    {
    public:
        friend SaveDataTransferManager;
        static const size_t InitialDataSize = InitialData::Size;

        /**
         * @brief インポート処理を確定します。
         *
         * @return 処理の結果を返します。
         *
         * @details インポートデータストリームへのデータ入力が全て完了していない時点で呼び出した場合、
         *          および入力されたデータの検証に失敗した場合は処理が失敗します。
         */
        Result Finalize() NN_NOEXCEPT;

        /*
         * @brief インポート対象のセーブデータ情報を取得します。
         *
         * @return 取得した情報の参照が返ります。
         */
        const SaveDataInfo& GetSaveDataInfo() NN_NOEXCEPT;

        /**
         * @brief インポートデータストリームにデータを入力し進めます。
         *
         * @return 処理の結果を返します。
         *
         * @pre Finalize が成功していない
         * @pre size <= GetRestSize()
         */
        Result Push(const void* buffer, size_t size) NN_NOEXCEPT;

        /*
         * @brief インポートデータストリームの残りのバイトサイズを取得します。
         *
         * @return 取得した値が返ります。
         */
        size_t GetRestSize() NN_NOEXCEPT;

    private:
        NN_IMPLICIT SaveDataImporter(sf::SharedPointer<fssrv::sf::ISaveDataImporter>&& interface) NN_NOEXCEPT;
        sf::SharedPointer<fssrv::sf::ISaveDataImporter> m_Interface;
        SaveDataInfo m_SaveDataInfo;
    };

    /**
     * @brief セーブデータ転送のセッションを構築するための管理クラスです。
     *
     * @details このクラスから取得した SaveDataExporter、SaveDataImporter は SetToken で設定されたトークンに基づいて転送データの暗復号化、署名検証を行います。
     *          インポート時に正しく転送データを検証・復号化するには、以下の条件を満たす必要があります。
     *          - エクスポート側とインポート側それぞれの SaveDataTransferManager に対となるトークンが設定されている
     *          - エクスポート側での SaveDataExporter の取得順序と、インポート側での SaveDataImporter の取得順序が一致したもの同士でデータ転送を行っている
     */
    class SaveDataTransferManager
    {
    public:
        static const size_t ChallengeSize = 16;

    public:
        SaveDataTransferManager() NN_NOEXCEPT;

        /*
         * @brief サーバーからトークンを取得するのに使う challenge を取得します。
         *
         * @return 処理の結果を返します。
         *
         * @pre outValue != nullptr && size >= ChallengeSize
         */
        Result GetChallenge(void* outValue, size_t size) NN_NOEXCEPT;

        /*
         * @brief トークンを設定します。
         *
         * @return 処理の結果を返します。
         *
         * @pre SetToken が成功していない
         * @pre saveDataTransferToken != nullptr
         */
        Result SetToken(const void* saveDataTransferToken, size_t size) NN_NOEXCEPT;

        /*
         * @brief SaveDataExporter のインスタンスを取得します。
         *
         * @return 処理の結果を返します。
         *
         * @pre SetToken が成功
         *
         * @details 対象のセーブデータが存在しない場合エラーが返ります。
         */
        Result OpenSaveDataExporter(std::unique_ptr<SaveDataExporter>* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT;

        /*
         * @brief SaveDataImporter のインスタンスを取得します。
         *
         * @return 処理の結果を返します。
         * @retval ResultUsableSpaceNotEnoughForSaveData インポートに必要な空き容量が足りません。
         *
         * @pre SetToken が成功
         *
         * @details 取得した SaveDataImporter は初期化データを生成した SaveDataExporter が対象としているセーブデータを、uid が指すアカウントの所有物としてインポートします。@n
         *          インポートに必要な空き容量の確認を行い、足りない場合は不足バイト数を outRequiredSize に格納し、ResultUsableSpaceNotEnoughForSaveData を返します。@n
         *          ResultUsableSpaceNotEnoughForSaveData 以外のエラーが返った場合 outRequiredSize の値は不定です。@n
         *          成功した場合、インポートに必要な容量を確保します。この確保した領域は SaveDataImporter::Finalize を呼ばずに SaveDataImporter を破棄すると消去されます。
         */
        Result OpenSaveDataImporter(std::unique_ptr<SaveDataImporter>* outValue, int64_t* outRequiredSize, const void* exportInitialData, size_t exportInitialDataSize, const account::Uid& uid, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

    public:
        Result OpenSaveDataImporter(std::unique_ptr<SaveDataImporter>* outValue, int64_t* outRequiredSize, const void* exportInitialData, size_t exportInitialDataSize, const fs::UserId& userId, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataTransferManager> m_Interface;
    };

//! @}

}}
