﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  セーブデータ管理 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ncm/ncm_ContentMetaId.h>

namespace nn { namespace fs {

//! @name セーブデータ管理 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づく各種セーブデータを作成します。
*
*   @param[out] pOutRequiredSize           容量不足だった場合、作成に必要な容量のバイトサイズ
*   @param[in]  applicationId              アプリケーションの識別子
*   @param[in]  applicationControlProperty アプリケーション管理データ
*   @param[in]  user                       ユーザーの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*        - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    指定されたアプリケーション管理データの情報から、アプリケーションに必要な各種セーブデータを作成します。@n
*               ユーザーアカウントセーブデータについては、user に指定されたユーザーアカウントのみが作成されます。
*               user に nn::account::InvalidUid が指定された場合、ユーザーアカウントセーブデータは作成されません。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、セーブデータ作成に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result EnsureApplicationSaveData(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, const account::Uid& user) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づく bcat データ配信キャッシュストレージを作成します。
*
*   @param[out] pOutRequiredSize           容量不足だった場合、作成に必要な容量のバイトサイズ
*   @param[in]  applicationId              アプリケーションの識別子
*   @param[in]  applicationControlProperty アプリケーション管理データ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*
*   @details    指定されたアプリケーション管理データの情報から、アプリケーションに必要な bcat データ配信キャッシュストレージを作成します。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、作成に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result EnsureApplicationBcatDeliveryCacheStorage(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      obsolete : 本 API はまもなく削除されます。
*               アプリケーションに紐づくキャッシュストレージを作成します。
*
*   @param[out] pOutRequiredSize           容量不足だった場合、作成に必要な容量のバイトサイズ容量不足だった場合、作成に必要な容量のバイトサイズ
*   @param[in]  applicationId              アプリケーションの識別子
*   @param[in]  applicationControlProperty アプリケーション管理データ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*
*   @details    指定されたアプリケーション管理データの情報から、アプリケーションに必要なキャッシュストレージを作成します。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、作成に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result EnsureApplicationCacheStorage(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づくキャッシュストレージを作成します。
*
*   @param[out] pOutRequiredSize             容量不足だった場合、作成に必要な容量のバイトサイズ
*   @param[out] pOutCacheStorageTargetMedia  キャッシュストレージを保存する対象メディア
*   @param[in]  applicationId                アプリケーションの識別子
*   @param[in]  applicationControlProperty   アプリケーション管理データ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*
*   @details    指定されたアプリケーション管理データの情報から、アプリケーションに必要なキャッシュストレージを作成します。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、作成に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result EnsureApplicationCacheStorage(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づくキャッシュストレージを作成します。
*
*   @param[out] pOutRequiredSize             容量不足だった場合、作成に必要な容量のバイトサイズ
*   @param[out] pOutCacheStorageTargetMedia  キャッシュストレージを保存する対象メディア
*   @param[in]  applicationId                アプリケーションの識別子
*   @param[in]  applicationControlProperty   アプリケーション管理データ
*   @param[in]  index                        キャッシュストレージのインデックス
*   @param[in]  cacheStorageSize             キャッシュストレージのサイズ
*   @param[in]  cacheStorageJournalSize      キャッシュストレージのジャーナルサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*
*   @details    指定されたアプリケーション管理データの情報から、アプリケーションに必要なキャッシュストレージを作成します。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、作成に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result CreateApplicationCacheStorage(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づく各種セーブデータを拡張します。
*
*   @param[out] pOutRequiredSize           容量不足だった場合、拡張に必要な容量のバイトサイズ
*   @param[in]  applicationId              アプリケーションの識別子
*   @param[in]  applicationControlProperty アプリケーション管理データ
*   @param[in]  saveDataType               セーブデータの種類
*   @param[in]  user                       ユーザーの識別子
*   @param[in]  saveDataSize               拡張後のセーブデータのデータ保存領域のサイズ
*   @param[in]  saveDataJournalSize        拡張後のセーブデータのジャーナリング領域のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultUsableSpaceNotEnough  空き容量が不足しています。
*
*   @pre
*        - pOutRequiredSize が有効なメモリを指している。
*        - saveDataType == nn::fs::SaveDataType::Account || saveDataType == nn::fs::SaveDataType::Device
*        - (static_cast\<bool\>(user) && saveDataType == nn::fs::SaveDataType::Account) || (!static_cast\<bool\>(user) && saveDataType == nn::fs::SaveDataType::Device)
*        - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    指定されたアプリケーション管理データの情報とサイズに基づいてセーブデータを拡張します。@n
*               拡張の必要がない場合は何もせず ResultSuccess を返します。@n
*               返り値に ResultUsableSpaceNotEnough が返った場合、セーブデータ拡張に必要な容量のバイトサイズが pOutRequiredSize に格納されます。@n
*/
Result ExtendApplicationSaveData(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, nn::fs::SaveDataType saveDataType, const account::Uid& user, int64_t saveDataSize, int64_t saveDataJournalSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      アプリケーションに紐づく各種セーブデータの現在のサイズを取得します。
*
*   @param[out] pOutSize                   セーブデータのデータ保存領域のサイズ
*   @param[out] pOutJournalSize            セーブデータのジャーナリング領域のサイズ
*   @param[in]  applicationId              アプリケーションの識別子
*   @param[in]  saveDataType               セーブデータの種類
*   @param[in]  user                       ユーザーの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*        - pOutSize が有効なメモリを指している。
*        - pOutJournalSize が有効なメモリを指している。
*        - saveDataType == nn::fs::SaveDataType::Account || saveDataType == nn::fs::SaveDataType::Device
*        - (static_cast\<bool\>(user) && saveDataType == nn::fs::SaveDataType::Account) || (!static_cast\<bool\>(user) && saveDataType == nn::fs::SaveDataType::Device)
*        - nn::account::Initialize() でアカウントシステムのライブラリが初期化済みである。
*
*   @details    アプリケーションに紐づくセーブデータのデータ保存領域およびジャーナリング領域のサイズを取得します。@n
*/
Result GetApplicationSaveDataSize(int64_t* pOutSize, int64_t* pOutJournalSize, ncm::ApplicationId applicationId, nn::fs::SaveDataType saveDataType, const account::Uid& user) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      一時ストレージを削除します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result CleanUpTemporaryStorage() NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
