﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/friends_TypesAdmin.h>
#include <nn/friends/friends_Macros.h>
#include <nn/friends/friends_ProfileExtra.h>

namespace nn { namespace friends {

/*!
    @brief      フレンドの詳細情報を扱うクラスです。
*/
class FriendDetailedInfo
{
public:
    /*!
        @brief      コンストラクタです。
    */
    FriendDetailedInfo() NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウント ID を取得します。

        @return     ネットワークサービスアカウント ID。
    */
    nn::account::NetworkServiceAccountId GetAccountId() const NN_NOEXCEPT;

    /*!
        @brief      最後のいっしょに遊んだ記録を取得します。

        @return     最後のいっしょに遊んだ記録。

        @details
                    最後のいっしょに遊んだ記録に保存されるのはアプリケーション情報とアプリ内表示名のみです。@n
                    最後にいっしょに遊んだ時刻は記録されないので注意してください。
    */
    const PlayRecord& GetLastPlayRecord() const NN_NOEXCEPT;

    /*!
        @brief      フレンド成立時のフレンド申請手段を取得します。

        @return     フレンド申請手段。
    */
    RequestType GetRequestType() const NN_NOEXCEPT;

    /*!
        @brief      フレンド成立時のフレンド申請の経路情報を取得します。

        @return     経路情報。

        @details
                    経路情報には、以下が含まれます。

                    - フレンド申請を行ったアプリケーションの情報@n
                      フレンド申請手段が @ref RequestType_InApp の場合のみ、有効な値が入ります。
                    - フレンド申請時の相手のアプリ内表示名@n
                      フレンド申請手段が @ref RequestType_InApp 、または、 @ref RequestType_FriendCandidate の場合、有効な値が入ります。
                    - フレンド成立した時刻
    */
    const RouteInfo& GetRequestRouteInfo() const NN_NOEXCEPT;

    /*!
        @brief      外部アプリケーションのカタログ ID を取得します。（フレンド候補専用）

        @return     外部アプリケーションのカタログ ID。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_FriendCandidate の場合のみ、有効な値が入ります。

                    本関数で取得したカタログ ID を使用することで、外部アプリケーションのカタログを取得することができます。

        @see
            - GetExternalApplicationCatalog
    */
    const ExternalApplicationCatalogId& GetExternalApplicationCatalogId() const NN_NOEXCEPT;

    /*!
        @brief      Mii の名前を取得します。（NNID フレンド専用）

        @return     Mii の名前。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    const MiiName& GetMiiName() const NN_NOEXCEPT;

    /*!
        @brief      Mii 画像 URL パラメータを取得します。（NNID フレンド専用）

        @return     Mii 画像 URL パラメータ。

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    const MiiImageUrlParam& GetMiiImageUrlParam() const NN_NOEXCEPT;

    /*!
        @brief      Mii 画像 URL を取得します。（NNID フレンド専用）

        @param[out] outUrl  Mii 画像 URL。

        @return     処理結果。

        @pre
            - outUrl != nullptr

        @details
                    本関数は、フレンド申請手段が @ref RequestType_3Ds 、または、 @ref RequestType_Nnid の場合のみ、有効な値が入ります。
    */
    nn::Result GetMiiImageUrl(Url* outUrl) const NN_NOEXCEPT;

    /*!
        @brief      拡張プロフィールを取得します。

        @return     拡張プロフィール。
    */
    const ProfileExtra& GetExtra() const NN_NOEXCEPT;

    /*!
        @brief      有効なデータを保持しているかどうかを確認します。

        @return     有効なデータを保持しているかどうか。
    */
    bool IsValid() const NN_NOEXCEPT;

private:
    //
    NN_FRIENDS_CLASS_STORAGE(2048);
};

}}
