﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

/*!
    @brief      JSON パスオブジェクトです。
*/
class JsonPath
{
public:
    /*!
        @brief      ノードの最大深さです。

        @details
                    ノードのネストは、ルート要素・オブジェクト・配列・キーを発見する度に発生します。@n
                    そのため、ノードの深さはネストの深さではないことに注意する必要があります。

                    例えば、 "$.array[0].data" のノードの深さは 5 になります。

                    - ルート要素 ($)
                    - キー名 (array)
                    - 配列 ([0])
                    - オブジェクト (array[0])
                    - キー名 (data)

                    ノードの最大深さを大きくすることにより、ネストの最大回数も増えます。@n
                    ネストの回数は関数の再帰呼び出し回数と一致するため、スタックオーバーフローに注意する必要があります。
    */
    static const int NodeDepthMax = 31;

    /*!
        @brief      JSON パス文字列の最大長です。

        @details
                    設定を変更する場合、 0xFFFF 以下にする必要があります。@n
                    パスのオフセット値を記録している @ref Node::head が uint16_t 型のための制限です。
    */
    static const int PathLengthMax = 255;

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonPath() NN_NOEXCEPT;

    /*!
        @brief      JSON パス文字列の比較を行います。

        @param[in]  path    JSON パス文字列。

        @pre
            - path != nullptr

        @details
                    本関数は、パスの完全一致比較を行います。@n
                    完全一致比較しか行わない場合、 @ref Match より高速に動作します。

                    階層の区切り文字は "." です。

                    JSON パスは以下のように記述します。

                    - ルート要素@n
                      "$" を記述します。
                    - キー@n
                      ".<key>" を記述します。
                    - 配列開始／終了@n
                      "[]" を記述します。
                    - 特定インデックスの配列要素@n
                      "[<index>]" を記述します。

                    例： {"array":[{"id": 0, "name": "A"}, {"id": 1, "name": "B"}]}

                    上記の 2 番目の id 要素を取得したい場合は、以下のように記述します。

                    - "$.array[1].id"
    */
    bool Compare(const char* path) const NN_NOEXCEPT;

    /*!
        @brief      JSON パス文字列のパターンマッチング比較を行います。

        @param[in]  path    JSON パス文字列。

        @pre
            - path != nullptr

        @details
                    本関数は、ワイルドカードを使用したパターンマッチングを行います。

                    階層の区切り文字は "." で、ワイルドカード文字は "*" です。@n
                    "*" は任意の 1 文字以上を意味します。

                    実装を簡略化するため、1 ノードに複数のワイルドカード文字は指定できません。@n
                    そのため、サポートするパターンマッチングは以下のようになります。

                    - 完全一致 (AB)
                    - 前方一致 (A*)
                    - 後方一致 (*B)
                    - 前後方一致 (A*B)

                    また、複数のノードを跨いだパターンマッチングはサポートしません。

                    JSON パスは以下のように記述します。

                    - ルート要素@n
                      "$" を記述します。
                    - キー@n
                      ".<key>" ／ ".<prefix>*" ／ ".*<postfix>" ／ ".<prefix>*<postfix>" のいずれかを記述します。
                    - 配列開始／終了@n
                      "[]" を記述します。
                    - 任意の配列要素@n
                      "[*]" を記述します。

                    例： {"array":[{"id": 0, "name": "A"}, {"id": 1, "name": "B"}]}

                    上記の id 要素をすべて取得したい場合は、以下のように記述します。

                    - "$.array[*].id"
    */
    bool Match(const char* path) const NN_NOEXCEPT;

    /*!
        @brief      JSON パス文字列を取得します。

        @return     JSON パス文字列。
    */
    const char* ToString() const NN_NOEXCEPT;

    /*!
        @brief      JSON パスが有効かどうかを判定します。

        @return     JSON パスが有効かどうかを判定します。

        @details
                    JSON パスのバッファが不足していた場合や、キー名が入力ストリームの文字列バッファに収まりきらなかった場合、無効になります。
    */
    bool IsValid() const NN_NOEXCEPT;

public:
    /*!
        @brief      オブジェクトノードの追加を試みます。

        @return     オブジェクトノードが追加できたかどうか。

        @details
                    本関数は、パーサー専用関数です。@n
                    ノードの深さが上限に達していた場合、false が返ります。
    */
    bool TryPushObject() NN_NOEXCEPT;

    /*!
        @brief      配列ノードの追加を試みます。

        @return     配列ノードが追加できたかどうか。

        @details
                    本関数は、パーサー専用関数です。@n
                    ノードの深さが上限に達していた場合、false が返ります。
    */
    bool TryPushArray() NN_NOEXCEPT;

    /*!
        @brief      キーノードの追加を試みます。

        @param[in]  key             キー。
        @param[in]  length          キーの長さ。
        @param[in]  isOverflowed    オーバーフローフラグ。

        @return     キーノードが追加できたかどうか。

        @details
                    本関数は、パーサー専用関数です。@n
                    ノードの深さが上限に達していた場合、false が返ります。
    */
    bool TryPushKey(const char* key, size_t length, bool isOverflowed) NN_NOEXCEPT;

    /*!
        @brief      現在のノードの情報を更新します。

        @details
                    本関数は、パーサー専用関数です。@n
                    現在のノードが配列ノードだった場合、要素数をインクリメントします。
    */
    void UpdateCurrent() NN_NOEXCEPT;

    /*!
        @brief      ノードを一つ取り出します。
    */
    void Pop() NN_NOEXCEPT;

    /*!
        @brief      現在のノードがキーノードだった場合、ノードを一つ取り出します。

        @return     現在のノードがキーノードだったかどうか。
    */
    bool PopIfKey() NN_NOEXCEPT;

private:
    /*!
        @brief      ノードの種類です。
    */
    enum NodeType : int8_t
    {
        NodeType_Root   = 0, //!< ルート要素。
        NodeType_Object = 1, //!< オブジェクト。
        NodeType_Array  = 2, //!< 配列。
        NodeType_Key    = 3  //!< キー。
    };

    /*!
        @brief      ノードです。
    */
    struct Node
    {
        NodeType type;       //!< ノード種別。
        bool validJsonPath;  //!< ノードの JSON パスが有効かどうか。
        uint16_t head;       //!< ノードの JSON パスの先頭位置。
        uint32_t arrayCount; //!< 配列の要素数。
    };

    NN_STATIC_ASSERT(PathLengthMax <= 0xFFFF);
    NN_STATIC_ASSERT(sizeof (Node) == 8);

private:
    //
    Node m_Nodes[NodeDepthMax + 1];
    //
    char m_Path[PathLengthMax + 1];
    //
    uint16_t m_Depth;
    //
    uint16_t m_PathLength;

private:
    //
    void InvalidateJsonPath() NN_NOEXCEPT;
};

}}}}}
