﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      ワークバッファです。

    @details
                大きいバッファが必要なタスクを実行する際に利用できるワークバッファです。

                ワークバッファを利用するには初めに @ref WorkBuffer::Acquire で所有権を取得する必要があります。@n
                ワークバッファの利用が完了した際は、必ず @ref WorkBuffer::Release で解放する必要があります。
*/
class WorkBuffer
{
private:
    NN_DISALLOW_COPY(WorkBuffer);
    NN_DISALLOW_MOVE(WorkBuffer);

public:
    /*!
        @brief      ワークバッファのサイズです。
    */
    static const size_t Size = 458 * 1024;

private:
    /*!
        @brief      コンストラクタです。
    */
    WorkBuffer() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static WorkBuffer& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(WorkBuffer, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      ワークバッファの所有権の取得を試みます。

        @return     ワークバッファの所有権を取得したかどうか。

        @details
                    他スレッドが所有権を確保していた場合、false が返ります。@n
                    しばらく待機した後、再度確保を試みてください。
    */
    bool TryAcquire() NN_NOEXCEPT;

    /*!
        @brief      ワークバッファの所有権を取得します。

        @return     ワークバッファの所有権を取得したかどうか。

        @details
                    本関数は、ワークバッファの所有権を取得するまでブロックします。@n
                    キャンセルされた場合のみ、false が返ります。
    */
    bool Acquire(const detail::service::util::Cancelable& cancelable) NN_NOEXCEPT;

    /*!
        @brief      ワークバッファの所有権を解放します。

        @pre
            - 所有権を確保している。
    */
    void Release() NN_NOEXCEPT;

    /*!
        @brief      ワークバッファを取得します。

        @return     ワークバッファ。

        @pre
            - 所有権を確保している。
    */
    void* Get() NN_NOEXCEPT;

    /*!
        @brief      ワークバッファのサイズを取得します。

        @return     ワークバッファのサイズ。
    */
    size_t GetSize() const NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    Bit64 m_Buffer[Size / 8];
};

// メモリ上にフレンド 300 人分の情報がすべて乗るかどうか。（SyncFriendList）
NN_STATIC_ASSERT(WorkBuffer::Size >= sizeof (FriendResource) * FriendCountMax);
// メモリ上にブロックしたユーザー 100 人分の情報がすべて乗るかどうか。（SyncBlockedUserList）
NN_STATIC_ASSERT(WorkBuffer::Size >= sizeof (BlockedUserResource) * BlockedUserCountMax);
// メモリ上にユーザー設定の情報がすべて乗るかどうか。（SyncUserSetting）
NN_STATIC_ASSERT(WorkBuffer::Size >= sizeof (UserResource));
// メモリ上に対面フレンド申請 50 件がすべて乗るかどうか。
NN_STATIC_ASSERT(WorkBuffer::Size >= sizeof (FacedFriendRequestResource) * FacedFriendRequestCountMax);
// メモリ上にプロフィール画像が乗るかどうか。
NN_STATIC_ASSERT(WorkBuffer::Size >= ProfileImageSizeMax);

}}}}}
