﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      ユーザープレゼンスの管理モジュールです。
*/
class UserPresenceManager
{
private:
    NN_DISALLOW_COPY(UserPresenceManager);
    NN_DISALLOW_MOVE(UserPresenceManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    UserPresenceManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static UserPresenceManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(UserPresenceManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      オンラインプレイの開始を宣言します。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。
    */
    nn::Result DeclareOpenOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      オンラインプレイの終了を宣言します。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。
    */
    nn::Result DeclareCloseOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザープレゼンスを設定します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  presence    ユーザープレゼンス。
        @param[in]  appInfo     アプリケーション情報。

        @return     処理結果。
    */
    nn::Result SetPresence(const nn::account::Uid& uid, const UserPresenceImpl& presence, const ApplicationInfo& appInfo) NN_NOEXCEPT;

    /*!
        @brief      ユーザープレゼンスを取得します。

        @param[in]  outPresence ユーザープレゼンス。
        @param[in]  uid         ユーザーアカウント。

        @return     処理結果。

        @pre
            - outPresence != nullptr
    */
    nn::Result GetPresence(UserPresenceViewImpl* outPresence, const nn::account::Uid& uid) const NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Open イベントを通知します。

        @param[in]  uid     ユーザーアカウント。
        @param[in]  appInfo ユーザーを Open したアプリケーションのアプリケーション情報。
    */
    void NotifyUserOpened(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Close イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      スリープを通知します。
    */
    void NotifySleep() NN_NOEXCEPT;

    /*!
        @brief      スリープ復帰を通知します。
    */
    void NotifySleepAwaked() NN_NOEXCEPT;

    /*!
        @brief      NPNS の接続状態の変化を通知します。

        @param[in]  isConnected 接続されているかどうか。
    */
    void NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        nn::account::Uid uid;
        UserPresenceViewImpl presence;
        bool isOpened;
        bool isOnlinePlaySessionOpened;
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    Record m_Records[nn::account::UserCountMax];
    bool m_IsNpnsConnected;

private:
    //
    int FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT;
    //
    void UpdateStatus(int index) NN_NOEXCEPT;
    void UpdateLastUpdateTime(int index) NN_NOEXCEPT;
};

}}}}}
