﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/core/friends_Task.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      フォアグラウンドタスクの管理モジュールです。
*/
class ForegroundTaskManager
{
private:
    NN_DISALLOW_COPY(ForegroundTaskManager);
    NN_DISALLOW_MOVE(ForegroundTaskManager);

public:
    /*!
        @brief      同時に実行可能なタスクの個数です。
    */
    static const int RunnableTaskCountMax = 2;

public:
    /*!
        @brief      コンストラクタです。
    */
    ForegroundTaskManager() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~ForegroundTaskManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ForegroundTaskManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ForegroundTaskManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      実行可能タスクの出現イベントを確保します。

        @return     実行可能タスクの出現イベント。

        @details
                    未使用のイベントを初期化して返します。@n
                    未使用のイベントが存在しない場合、ASSERT します。

                    本関数は、タスクを実行するスレッド 1 つに付き、1 回だけ呼び出すようにしてください。
    */
    nn::os::EventType* AcquireRunnableTaskEvent() NN_NOEXCEPT;

    /*!
        @brief      実行可能タスクの出現イベントを解放します。

        @param[in]  runnableEvent   出現イベント。

        @details
                    @ref AcquireRunnableTaskEvent で確保したイベントを開放します。
    */
    void ReleaseRunnableTaskEvent(nn::os::EventType* runnableEvent) NN_NOEXCEPT;

    /*!
        @brief      タスクをタスクキューに追加します。

        @param[in]  task    タスク。

        @pre
            - task != nullptr
    */
    void Add(Task* task) NN_NOEXCEPT;

    /*!
        @brief      タスクをタスクキューから削除します。

        @param[in]  task    タスク。

        @pre
            - task != nullptr

        @details
                    タスクが自動削除モードだった場合、タスク自体の削除も行います。
    */
    void Remove(Task* task) NN_NOEXCEPT;

    /*!
        @brief      すべてのタスクをタスクキューから削除します。

        @details
                    タスクが自動削除モードだった場合、タスク自体の削除も行います。
    */
    void RemoveAll() NN_NOEXCEPT;

    /*!
        @brief      実行可能なタスクを 1 つ実行します。

        @return     タスクを実行したかどうか。

        @details
                    実行可能なタスクを検索し、実行します。@n
                    実行可能なタスクが存在しない場合、即座に処理を返します。
    */
    bool RunRunnable() NN_NOEXCEPT;

private:
    /*!
        @brief      タスクイベントです。
    */
    struct TaskEvent
    {
        nn::os::EventType runnableEvent; //!< 実行可能タスク出現イベント。
        bool allocated;                  //!< 確保済みフラグ。
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    TaskEvent m_TaskEvents[RunnableTaskCountMax];
    //
    nn::util::IntrusiveList<Task, nn::util::IntrusiveListBaseNodeTraits<Task>> m_TaskQueue;

private:
    //
    bool HasRunnable() NN_NOEXCEPT;
    //
    void SignalRunnableTaskEvents() NN_NOEXCEPT;
};

}}}}}
