﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>
#include <nn/friends/detail/service/core/friends_FileSystemConfig.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      アカウント別ストレージの管理モジュールです。
*/
class AccountStorageManager
{
private:
    NN_DISALLOW_COPY(AccountStorageManager);
    NN_DISALLOW_MOVE(AccountStorageManager);

public:
    /*!
        @brief      アカウント別ストレージのサイズです。

        @details
                    内訳は以下の通りです。

                    - ユーザー設定キャッシュ
                    -- friends:/<nsaid>/setting.cache
                    -- 2KB
                    -- 1 block: 16KB
                    - フレンドリストキャッシュ
                    -- friends:/<nsaid>/friend.cache
                    -- MAX 450KB
                    -- 29 blocks: 464KB
                    - ブロックリストキャッシュ
                    -- friends:/<nsaid>/block.cache
                    -- MAX 31KB
                    -- 2 blocks: 32KB
                    - 対面フレンド申請リスト
                    -- friends:/<nsaid>/faced.bin
                    -- MAX 11KB
                    -- 1 block: 16KB
                    - 対面フレンド申請プロフィール画像
                    -- friends:/<nsaid>/faced/<nsaid>.jpg
                    -- MAX 128KB * 50
                    -- 400 blocks: 6400KB
                    - フレンドプロフィール画像ダウンロード要求キュー
                    -- friends:/<nsaid>/image.queue
                    -- MAX 3KB
                    -- 1 block: 16KB
                    - プレイログ
                    -- friends:/<nsaid>/playlog.bin
                    -- 1KB
                    -- 1 block: 16KB
                    - いっしょに遊んだ記録
                    -- friends:/history.bin
                    -- MAX 75KB
                    -- 5 blocks: 80KB
                    - いっしょに遊んだ記録の統計情報
                    -- friends:/history.stats.bin
                    -- 1KB
                    -- 1 block: 16KB
                    - UUID
                    -- friends:/uuid.bin
                    -- 1KB
                    -- 1 block: 16KB
                    - エントリー情報
                    -- 2 blocks: 32KB

                    合計：444 blocks: 7104KB

                    設定値：464 blocks: 7424KB (reserved 20 blocks: 320KB)
    */
    static const int64_t StorageSize = 464 * BlockSize;

    /*!
        @brief      アカウント別ストレージのジャーナルサイズです。

        @details
                    一番ジャーナルを消費するアクセスは以下の通りです。

                    - フレンドキャッシュの作成
                    -- 前のキャッシュデータ削除：1
                    -- キャッシュデータ作成：1
                    -- キャッシュデータ書込：29

                    合計：31 blocks: 496KB

                    設定値：48 blocks: 768KB (reserved 17 blocks: 272KB)
    */
    static const int64_t StorageJournalSize = 48 * BlockSize;

private:
    /*!
        @brief      コンストラクタです。
    */
    AccountStorageManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static AccountStorageManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(AccountStorageManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      システムセーブデータ ID を設定します。

        @param[in]  id  システムセーブデータ ID。

        @details
                    初期値は FS チームより割り当てられた 0x8000000000000080 が設定されています。@n
                    テスト等でシステムセーブデータ ID を変更したい場合、本関数を使用してください。
    */
    void SetSystemSaveDataId(nn::fs::SystemSaveDataId id) NN_NOEXCEPT;

    /*!
        @brief      アカウント別ストレージをマウントします。

        @param[in]  uid                 ユーザーアカウント。
        @param[in]  createIfNotExists   ストレージが存在しない時に作成するかどうか。

        @return     処理結果。

        @details
                    マウントを行うと、呼び出しスレッドが所有権を確保します。@n
                    別スレッドが所有権を確保している場合、解放されるまで本関数はブロックします。
    */
    nn::Result Mount(const nn::account::Uid& uid, bool createIfNotExists = true) NN_NOEXCEPT;

    /*!
        @brief      アカウント別ストレージをアンマウントします。

        @pre
            - 所有権を確保している。
    */
    void Unmount() NN_NOEXCEPT;

    /*!
        @brief      アカウント別ストレージをコミットします。

        @return     処理結果。

        @pre
            - 所有権を確保している。

        @details
                    アカウント別ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    nn::Result Commit() NN_NOEXCEPT;

    /*!
        @brief      管理外のネットワークサービスアカウントディレクトリを削除します。

        @return     処理結果。

        @pre
            - 所有権を確保している。

        @details
                    ネットワークサービスアカウントの紐付けが変更される等して、管理外となったネットワークサービスアカウントのディレクトリを削除します。
    */
    nn::Result DeleteUnmanagedNetworkServiceAccountDirectory() NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウントディレクトリを削除します。

        @return     処理結果。

        @pre
            - 所有権を確保している。
    */
    nn::Result DeleteNetworkServiceAccountDirectory() NN_NOEXCEPT;

    /*!
        @brief      アカウント別ストレージ内のファイルにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  name    ファイル名。

        @pre
            - path != nullptr
            - size > 0
            - name != nullptr
            - 所有権を確保している。
    */
    void MakePath(char* path, size_t size, const char* name) NN_NOEXCEPT;

    /*!
        @brief      アカウント別ストレージ内のネットワークサービスアカウントに紐付くファイルにアクセスするためのパスを作成します。

        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  name    ファイル名。

        @return     処理結果。
        @retval     ResultNetworkServiceAccountNotLinked    ネットワークサービスアカウントが紐付いていない。

        @pre
            - path != nullptr
            - size > 0
            - name != nullptr
            - 所有権を確保している。
    */
    nn::Result MakePathWithNetworkServiceAccountDirectory(char* path, size_t size, const char* name) NN_NOEXCEPT;

public:
    /*!
        @brief      対面フレンド申請のプロフィール画像 URL を作成します。

        @param[in]  url         バッファ。
        @param[in]  size        バッファサイズ。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   自分のネットワークサービスアカウント ID。
        @param[in]  accountId   取得したい相手のネットワークサービスアカウント ID。

        @pre
            - path != nullptr
            - size > 0
    */
    static void MakeFacedFriendRequestProfileImageUrl(char* url, size_t size, const nn::account::Uid& uid,
        nn::account::NetworkServiceAccountId myAccountId, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請のプロフィール画像 URL の名前解決を行います。

        @param[out] outUid  対象のユーザーアカウント。
        @param[in]  path    バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  url     URL。

        @return     名前解決できたかどうか。

        @pre
            - outUid != nullptr
            - path != nullptr
            - size > 0
            - url != nullptr
    */
    static bool ResolveFacedFriendRequestProfileImageUrl(nn::account::Uid* outUid,
        char* path, size_t size, const char* url) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_Uid;
    //
    nn::account::NetworkServiceAccountId m_AccountId;
    bool m_IsNetworkServiceAccountAvailable;
    //
    nn::fs::SystemSaveDataId m_SystemSaveDataId;
};

}}}}}

/*!
    @brief      アカウント別ストレージを局所的にロックします。
*/
#define NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid) \
    NN_RESULT_DO(detail::service::core::AccountStorageManager::GetInstance().Mount(uid));   \
    NN_UTIL_SCOPE_EXIT                                                                      \
    {                                                                                       \
        detail::service::core::AccountStorageManager::GetInstance().Unmount();              \
    }

/*!
    @brief      アカウント別ストレージを局所的にロックします。

    @details
                ストレージが存在しない場合、nn::fs::ResultTargetNotFound が返されます。
*/
#define NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK_IF_EXISTS(uid) \
    NN_RESULT_DO(detail::service::core::AccountStorageManager::GetInstance().Mount(uid, false));    \
    NN_UTIL_SCOPE_EXIT                                                                              \
    {                                                                                               \
        detail::service::core::AccountStorageManager::GetInstance().Unmount();                      \
    }
