﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/fgm/fgm_ModuleId.h>
#include <nn/fgm/fgm_Result.h>

namespace nn     {
namespace fgm    {

// Number of priorities supported by FGM
const int  NumberOfPriorities  = 3;

/**
    @brief      FGM request priority.
*/
typedef uint32_t Priority;

/**
    @brief Definitions for nn::fgm::Priority
 */
enum PriorityDefinitions
{
    /*
        @brief Highest priority, overrides all other requests.
    */
    Priority_Highest = 0,

    Priority_Thermal = Priority_Highest,

    /**
        @brief Default priority. SDK users should use this value.
    */
    Priority_Default = 8,

    Priority_Lowest  = 9,
};

/**
    @internal
    @brief      FGM priorities for each port.
*/
const PriorityDefinitions FgmPortPriority[NumberOfPriorities] = {
    Priority_Highest,
    Priority_Default,
    Priority_Lowest
};

/**
    @internal
    @brief      FGM debugger port name.
*/
const char FgmDebugPortName[]      = "fgm:dbg";

/**
    @internal
    @brief      FGM port names for each priority.
*/
const char* const PortName[NumberOfPriorities] = {
    // Highest priority port name
    "fgm:0",
    // Default priority port name
    "fgm",
    // Lowest priority port name
    "fgm:9",
};

/**
    @brief      The type to express frequency of each module.
*/
typedef uint32_t Setting;

/**
    @brief Definitions for nn::fgm::Setting
*/
enum SettingDefinitions
{
    /**
        @brief      Minimum @ref Setting value supported by each module.
        @details    When client wants to limit operating range to the minimum setting
                    that is supported by the module, it can pass Setting_Min in place
                    of the actual value. FGM will replace Setting_Min with the actual
                    minimum value that module supports.
    */
    Setting_Min = 0,

    /**
        @brief      Maximum @ref Setting supported by each module.
        @details    When client wants to limit operating range to the maximum setting
                    that is supported by the module, it can pass Setting_Max in place
                    of the actual value. FGM will replace Setting_Max with the actual
                    maximum value that module supports.
    */
    Setting_Max = UINT_MAX,
};

/**
    @internal
    @brief      Maximum number of simultanous FGM requests for each priority.
    @details    Each process has a limit on the number of simultanious requests
                at each priority. When this limit is exceeded, an error will be
                returned from Initialize() call.
*/
const int MaximumRequestsPerPriority = 40;

}}
