﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
    @file
    @brief  Public header defining FGM API.
 */

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fgm/fgm_Types.h>
#include <nn/fgm/sfdl/fgm.sfdl.h>

namespace nn  {
namespace fgm {

/**
    @brief      Request for changing state of each module.

    @details    To request a new operating range FGM client should allocate and
                initialize this class. Once initialized, this class can be used
                to set new min-max range on the module, update existing
                request with new min-max range, or obtain notification when
                module setting is updated due to other requests in the system.

                Request will stay active until client calls Finalize() or
                destroys this object.

                Sample code:
                @code
                #include <nn/fgm.h>
                nn::fgm::Request request;
                void test(Setting min, Setting max)
                {
                    nn::fgm::Setting current;
                    if (request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default).IsFailure() ||
                        request.SetAndWait(min, max).IsFailure() ||
                        request.Get(&current).IsFailure())
                    {
                        // handle failure;
                    }
                    // once no longer needed
                    request.Finalize();
                }
                @endcode
*/
class Request
{
    NN_DISALLOW_COPY(Request);
    NN_DISALLOW_MOVE(Request);

private:
    nn::sf::SharedPointer<nn::fgm::sf::IRequest> m_RequestImpl;
    nn::os::SystemEvent m_SystemEvent;
    int m_PortIndex;

    // Once this header becomes public, we will not be
    // able to modify size of this class.
    // Leave uintptr_t so that if need arises we'll be able
    // to add a reference to a chunk allocated by fgm
    // library to support new additions.
    uintptr_t           m_Reserved;

public:
    /**
        @brief      Initialize this object.
        @param[in]  id                                  Target module id as defined in fgm_Types.h.
        @param[in]  priority                            Priority of the request.
                                                        SDK users should specify
                                                        @ref nn::fgm::Priority_Default for this parameter.
                                                        Other values are for internal use.
        @param[in]  eventClearMode                      Clear mode of notification system event.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultModuleNotFound       Unsupported module id.
        @retval     nn::fgm::ResultInvalidPriority      Unsupported priority.
        @retval     nn::Result                          Other failure code.

        @details    The Initialize() function is used to initialize FGM request. This function
                    ties request to a specific module with priority.
                    Parameter @a eventClearMode may be used to select event signaling mode.
    */
    nn::Result Initialize(Module id, Priority priority, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
        @brief      Initialize this object.
        @param[in]  id                                  Target module id as defined in fgm_Types.h.
        @param[in]  priority                            Priority of the request.
                                                        SDK users should specify
                                                        @ref nn::fgm::Priority_Default for this parameter.
                                                        Other values are for internal use.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultAlreadyInitialized   Already initialzied.
        @retval     nn::fgm::ResultModuleNotFound       Unsupported module id.
        @retval     nn::fgm::ResultInvalidPriority      Unsupported priority.
        @retval     nn::sf::ResultMemoryAllocationFailed Current process has already
                                                         initialized MaximumRequestsPerPriority requests
                                                         using this priority.
        @retval     nn::Result                          Other Failure code.

        @details    See @ref Initialize() for description. This function will set up event with manual clear mode.
                    If auto clear mode is desired, please use Initialize function that takes
                    event clear mode as argument.
    */
    nn::Result Initialize(Module id, Priority priority) NN_NOEXCEPT;

    /**
        @brief      Request new operating range of frequency.
        @param[in]  min                                 Minimum value of operating range.
        @param[in]  max                                 Maximum value of operating range.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultInvalidRange         Requested @a min is greater than requested @a max.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::fgm::ResultNotOperational       FGM not operational in current PM state
        @retval     nn::Result                          Other failure code.

        @details    The Set() function may be used to set, or update existing request,
                    with new desired operating range.
                    As a result of the change operating range may change its actual frequency.
                    Both @a min and @a max may have the same value.

                    Note that this function does not wait for actual state transition
                    to complete. Clients may choose to use @ref Wait() or @ref WaitWithTimeout()
                    functions to detect completion of state transition. Alternatively,
                    clients may obtain reference to system event using @ref GetSystemEventPointer(),
                    and use one of the mechanisms provided by OS to block on an event.
    */
    nn::Result Set(Setting min, Setting max) NN_NOEXCEPT;

    /**
        @brief      Request new operating range of frequency and wait for module to complete transition.
        @param[in]  min                                 Minimum value of operating range.
        @param[in]  max                                 Maximum value of operating range.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultInvalidRange         Requested @a min is greater than requested @a max.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::fgm::ResultNotOperational       FGM not operational in current PM state
        @retval     nn::Result                          Other failure code.

        @details    See @ref Set() for description. This function differs from @ref Set in that
                    it will block waiting for module to complete transition into new state.
    */
    nn::Result SetAndWait(Setting min, Setting max) NN_NOEXCEPT;

    /**
        @brief      Request new operating range and wait with timeout for module to complete transition.
        @param[in]  min                                 Minimum value of operating range.
        @param[in]  max                                 Maximum value of operating range.
        @param[in]  timeout                             Time to wait for transition completion into new frequency.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed succesfully.
        @retval     nn::fgm::ResultInvalidRange         Requested @a min is greater than requested @a max.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::fgm::ResultTimeOut              Operation timed out.
        @retval     nn::fgm::ResultNotOperational       FGM not operational in current PM state
        @retval     nn::Result                          Other failure code.

        @details    See @ref Set() for description. This function will block waiting for
                    module to complete transition into new frequency. This function will
                    return timeout error if completion event is not signaled within the
                    timeout period.
    */
    nn::Result SetAndWaitWithTimeout(Setting min, Setting max, nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
        @brief      Obtain current frequency of the module.
        @param[out] pCurrent                            Location storing the current actual frequency.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::fgm::ResultNotOperational       FGM not operational in current PM state
        @retval     nn::Result                          Other failure code.

        @details    The Get() function may be used to obtain current actual frequncy of the module
                    associated with this request.
    */
    nn::Result Get(Setting* pCurrent) NN_NOEXCEPT;

    /**
        @brief      Wait for event notifications.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::Result                          Other failure code.

        @details    The Wait() function may be used to block waiting for event indicating
                    change of module frequency. The change may happen as a
                    result of @ref Set() operation from this client or as a result of
                    requests from other users of FGM in the system. When event notification is
                    detected, client is advised to read state of the module,
                    and, if possible, adjust its operations accordingly - for example
                    if GPU clock went down, adjust frame rate, etc...

                    Before returning, this function will manually clear event.
    */
    nn::Result Wait() NN_NOEXCEPT;

    /**
        @brief      Wait for event notifications with timeout.
        @param[in]  timeout                             Timespan class defining timeout.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultTimeOut              Operation timed out.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::Result                          Other failure code.

        @details    The WaitWithTimeout() function is similar to @ref Wait(),
                    but it takes @a timeout argument.

                    Before returning, this function will manually clear event.
    */
    nn::Result WaitWithTimeout(nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
        @brief      Cancel FGM request.

        @return                                         Result of API execution.
        @retval     nn::ResultSuccess                   Operation completed successfully.
        @retval     nn::fgm::ResultNotInitialized       This @ref Request object has not been initialized.
        @retval     nn::Result                          Failure code.

        @details    The Cancel() function may be used to abort pending operations.
                    This call will change state of the event to 'signaled'. This
                    call may be used to unblock @ref Wait() or @ref WaitWithTimeout()
                    calls.
    */
    nn::Result Cancel() NN_NOEXCEPT;

    /**
        @brief      Obtain handle to system event associated with this request.
        @return     Pointer to system event.

        @details    Clients wishing to implement event handling logic may obtain
                    pointer to the system event associated with this request.
                    By default system event is initialized in manual clear mode.
                    If auto clear mode is desired, please use @ref Initialize()
                    function that accepts an argument for event clear mode.
    */
    nn::os::SystemEvent* GetSystemEventPointer() NN_NOEXCEPT;

    /**
        @brief      Clear system event manually.
        @details    The ClearEvent() function may be used to clear system event manually.
                    The system event can be obtained by GetSystemEventPointer().
    */
    void ClearEvent() NN_NOEXCEPT;

    /**
        @brief      Finalize this object.

        @return                                     Result of API execution.
        @retval     nn::ResultSuccess()             Operation completed successfully.

        @details    The Finalize() call may be used to de-associated this object from the module.
    */
    nn::Result Finalize() NN_NOEXCEPT;

    Request()  NN_NOEXCEPT;
    ~Request() NN_NOEXCEPT;
};

}}
