﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService に関する型や定数の宣言
*/

#pragma once

#include <cstring>
#include <nn/fs/fs_RightsId.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace es {

/*!
* @brief TicketIdを表す数値です。
*/
typedef uint64_t TicketId;

/*!
* @brief DeviceIdを表す数値です。
*/
typedef uint64_t DeviceId;

/*!
* @brief RightsIdを表す数値です。
*/
typedef uint64_t RightsId;

/*!
* @brief KeyIdを表す数値です。
*/
typedef uint64_t KeyId;

/*!
* @brief KeyIdを含んだ128bit RightsIdを表す数値です。
*/
struct RightsIdIncludingKeyId
{
    uint8_t _data[16];

    bool operator==(const RightsIdIncludingKeyId& rhs) const NN_NOEXCEPT
    {
        return (memcmp(this->_data, rhs._data, sizeof(this->_data)) == 0);
    }

    bool operator!=(const RightsIdIncludingKeyId& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }

    bool operator<(const RightsIdIncludingKeyId& rhs) const NN_NOEXCEPT
    {
        for (int i = 0; i < sizeof(this->_data); i++)
        {
            if (this->_data[i] < rhs._data[i])
            {
                return true;
            }
            else if (this->_data[i] > rhs._data[i])
            {
                return false;
            }
        }

        return false;
    }

    RightsId GetRightsId() const NN_NOEXCEPT
    {
        return util::LoadBigEndian<RightsId>(reinterpret_cast<const RightsId*>(&this->_data[0]));
    }

    KeyId GetKeyId() const NN_NOEXCEPT
    {
        return util::LoadBigEndian<KeyId>(reinterpret_cast<const KeyId*>(&this->_data[sizeof(RightsId)]));
    }

    bool IsExternalKey() const NN_NOEXCEPT
    {
        const RightsIdIncludingKeyId zeroRightsId = {};
        return *this != zeroRightsId;
    }

    static RightsIdIncludingKeyId Construct(const nn::fs::RightsId& rhs) NN_NOEXCEPT
    {
        RightsIdIncludingKeyId rightsId;
        std::memcpy(rightsId._data, rhs.data, sizeof(rightsId._data));
        return rightsId;
    }
};
NN_STATIC_ASSERT(sizeof(::nn::es::RightsIdIncludingKeyId) == sizeof(::nn::fs::RightsId));

/*!
* @brief AccountIdを表す数値です。
*/
typedef uint32_t AccountId;

/*!
* @brief チケットのバージョンを表す数値です。
*/
typedef uint16_t TicketVersion;

/*!
* @brief AesKeyを表す数値です。
*/
struct AesKey
{
    uint8_t _data[16];
};

/*!
* @brief RsaKeyを表す数値です。
*/
struct RsaKey
{
    uint8_t _data[256];
};

/*!
* @brief 署名データを表す数値です。
*/
struct Sign
{
    uint8_t _data[60];
};

/*!
* @brief 証明書データを表す数値です。
*/
struct Certificate
{
    uint8_t _data[384];
};

/*!
* @brief 鍵タイプを表す数値です。
*/
enum KeyType {
    KeyType_AesKey, //!< Aes
    KeyType_RsaKey, //!< Rsa
};

/*!
* @brief 鍵を表す構造体です。
*/
struct TitleKey
{
    /*!
    * @brief 鍵タイプを表す数値です。
    */
    KeyType keyType;

    /*!
    * @brief 鍵の数値です。
    */
    union {
        AesKey aesKey;
        RsaKey rsaKey;
    } key;
};

/*!
* @brief プロパティマスクのフラグです。
*/
enum class PropertyMaskFlag : uint16_t
{
    // PreInstall = 0x1,        // NX では使用していない
    // SharedTitle = 0x2,       // NX では使用していない
    // AllContents = 0x4,       // NX では使用していない
    IsElicenseRequired = 0x8,
};

/*!
* @brief チケットのデータを示す構造体です。
*/
struct TicketInfo
{
    /*!
    * @brief TicketIdです。
    */
    TicketId ticketId;

    /*!
    * @brief DeviceIdです。
    */
    DeviceId deviceId;

    /*!
    * @brief RightsIdです。
    */
    RightsIdIncludingKeyId rightsId;

    /*!
    * @brief AccountIdです。
    */
    AccountId accountId;

    /*!
    * @brief チケットのサイズです。
    */
    int64_t ticketSize;

    /*!
    * @brief チケットのバージョンです。
    */
    TicketVersion ticketVersion;

    /*!
    * @brief プロパティマスクです。
    */
    uint16_t propertyMask;

    /*!
    * @brief パディングです。
    */
    int32_t padding;

    /*!
    * @brief 将来の拡張のための予約領域です。
    */
    int64_t reserved;

    /*!
    * @brief 利用するために eLicense が必要なチケットかどうかを判定します。
    */
    bool IsElicenseRequired() const NN_NOEXCEPT
    {
        return (propertyMask & static_cast<std::underlying_type<PropertyMaskFlag>::type>(PropertyMaskFlag::IsElicenseRequired)) != 0;
    }
};

struct LightTicketInfo
{
    /*!
    * @brief RightsIdです。
    */
    RightsIdIncludingKeyId rightsId;

    /*!
    * @brief TicketIdです。
    */
    TicketId ticketId;

    /*!
    * @brief AccountIdです。
    */
    AccountId accountId;

    /*!
    * @brief プロパティマスクです。
    */
    uint16_t propertyMask;

    /*!
    * @brief パディングです。
    */
    int16_t padding;
};

struct PrepurchaseRecord
{
    /*!
    * @brief RightsIdです。
    */
    RightsIdIncludingKeyId rightsId;

    /*!
    * @brief TicketIdです。
    */
    TicketId ticketId;

    /*!
    * @brief AccountIdです。
    */
    AccountId accountId;

    /*!
    * @brief 予約領域です。
    */
    int8_t reserved[28];

    /*!
    * @brief 配信開始予定時刻です。
    */
    time::PosixTime deliveryScheduledTime;
};
NN_STATIC_ASSERT(sizeof(PrepurchaseRecord) == 64);

}}
