﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService の eLicense に関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/es/es_ELicenseApi.h>
#include <nn/es/es_ELicenseTypesForInner.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace es {

/**
*   @brief      デバッグ用に Payload のみの eLicense アーカイブのインポートを完了します。
*
*   @param[out] outELicenseArchiveId            取得した ELicenseArchiveId
*   @param[in]  context                         コンテキスト
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @details    Payload のフォーマットは Json 形式である必要があります。
*               本 API でインポートされた eLicense アーカイブは本体ストレージに保存されず、再起動後は無効化されます。
*               本 API は製品機では使用できません。
*               製品機で本 API を呼び出した場合は関数の中でアボートします。
*/
Result EndImportELicenseArchiveForDebug(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT;

/**
*   @brief      デバッグ用に eLicense アーカイブの Payload のサイズを取得します。
*
*   @param[out] pOutSize                        取得した eLicense アーカイブの Payload のサイズ
*   @param[in]  naId                            取得する eLicense アーカイブの NintendoAccountId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultELicenseArchiveNotFound   ELicense アーカイブが存在しませんでした。
*
*   @details    本 API は製品機では使用できません。
*               製品機で本 API を呼び出した場合は関数の中でアボートします。
*/
Result GetELicenseArchiveSizeForDebug(size_t* pOutSize, account::NintendoAccountId naId) NN_NOEXCEPT;

/**
*   @brief      デバッグ用に eLicense アーカイブの Payload のデータを取得します。
*
*   @param[out] pOutSize                        取得した eLicense アーカイブの Payload のサイズ
*   @param[out] pOutBuffer                      取得した eLicense アーカイブを格納するバッファ
*   @param[in]  size                            取得した eLicense アーカイブを格納するバッファのサイズ
*   @param[in]  naId                            取得する eLicense アーカイブの NintendoAccountId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultELicenseArchiveNotFound   ELicense アーカイブが存在しませんでした。
*   @retval     ResultBufferNotEnough           バッファが不足しています。
*
*   @details    本 API は製品機では使用できません。
*               製品機で本 API を呼び出した場合は関数の中でアボートします。
*/
Result GetELicenseArchiveDataForDebug(size_t* pOutSize, void* pOutBuffer, size_t size, account::NintendoAccountId naId) NN_NOEXCEPT;

/**
*   @brief      デバッグ用に eLicense の情報を取得します。
*
*   @param[out] outELicenseInfoList[]           取得した eLicense の情報を格納する配列
*   @param[in]  eLicenseInfoCount               取得した eLicense の情報を格納する配列の要素数
*   @param[in]  eLicenseIdList[]                eLicense の情報を取得する ELicenseId の配列
*   @param[in]  eLicenseIdCount                 eLicense の情報を取得する ELicenseId の配列の要素数
*
*   @return     eLicense の情報を取得した数
*
*   @details    本 API は製品機では使用できません。
*               製品機で本 API を呼び出した場合は関数の中でアボートします。
*/
int ListELicenseInfoForDebug(ELicenseInfoForSystemWrapper outELicenseInfoList[], int eLicenseInfoCount, const ELicenseId eLicenseIdList[], int eLicenseIdCount) NN_NOEXCEPT;

/**
*   @brief      デバッグ用に eLicense の情報と保存されている eLicense アーカイブを削除します。
*               本 API の呼出し後は eLicense により権利が管理されていない状態になります。
*
*   @details    本 API は製品機では使用できません。
*               製品機で本 API を呼び出した場合は関数の中でアボートします。
*/
void DeleteAllELicenseArchiveForDebug() NN_NOEXCEPT;

}}

