﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_CalendarTime.h>

namespace nn { namespace err {
/**
* @brief    サービスステータス情報を表すクラスです。
*/
class ServiceStatus
{
public:
    bool                   hasMaintenance;       //!< メンテナンスがあるかどうかのフラグ。
    nn::time::CalendarTime startMaintenanceDate; //!< メンテナンス開始時刻（UTC）。必ず存在する。
    nn::time::CalendarTime endMaintenanceDate;   //!< メンテナンス終了時刻（UTC）。必ず存在する。

    bool                   hasIncident;          //!< 障害情報があるかどうかのフラグ。
    nn::time::CalendarTime startIncidentDate;    //!< 障害発生時刻（UTC）。必ず存在する。
    nn::time::CalendarTime endIncidentDate;      //!< 障害復旧時刻（UTC）。存在しない場合がある。
};

/**
* @brief    サービスステータス情報にアクセスするクラスです。
*/
class ServiceStatusAccessor
{
public:
    /**
    * @brief    ServiceStatusAccessor クラスのインスタンスを作成します。
    */
    ServiceStatusAccessor() NN_NOEXCEPT
    {
        m_Data = NULL;
        m_Size = 0;
    };

    /**
    * @brief        ServiceStatusAccessor クラスの初期化を行います。
    * @param[in]    data  サービスステータス情報を取得する Json 形式のデータ(NULL終端付き)。
    * @param[in]    size  データサイズ。
    */
    void Initialize(void* data, size_t size) NN_NOEXCEPT;

    /**
    * @brief        サービスステータス情報を取得します。
    * @param[out]   serviceStatus  サービスステータス情報を格納する ServiceStatus クラスへのポインタ。
    * @param[in]    applicationId  サービスステータス情報取得対象のアプリケーションID。
    * @param[in]    serverCode     サービスステータス情報取得対象のサーバーコード。
    * @retresult
    *    @handleresult{nn::ResultSuccess,                       取得に成功しました。}
    *    @handleresult{nn::err::ResultServiceStatusParseError,  サービスステータスの解析に失敗しました。}
    * @endretresult
    */
    nn::Result GetServiceStatus(ServiceStatus* serviceStatus, const nn::ApplicationId& applicationId, const char* serverCode) const NN_NOEXCEPT;

private:
    void*           m_Data;
    size_t          m_Size;
};

}}  // namespace nn::err
