﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/err/err_Types.h>
#include <nn/ns/ns_ApplicationManagerApi.h>

namespace nn { namespace err {

/**
 * @brief エラーの情報を表すクラスです。
 */
class ErrorInfo
{
public:

    /**
    * @brief エラービューアアプレットの画面表示に関する情報を表す構造体です。
    */
    struct UiSettings
    {
        static const int ButtonCountMax = 3;                                //!< エラービューアアプレットが表示するボタンの最大数です。
        static const int ButtonMessageLengthMax = 128 / sizeof(char16_t);   //!< エラービューアアプレットのボタン上に表示するメッセージ文字列配列の長さです。NULL終端の分を含めています。
        static const int MessageLengthMax = 2048;                           //!< エラービューアアプレットのメッセージ文字列配列の長さです。NULL終端の分を含めています。

        int         messageLength;                                          //!< NULL終端を除いたメッセージの長さです。message[messageLength - 1] まで有効な情報が含まれています。
        uint8_t     buttonCount;                                            //!< ボタン数です。
        uint8_t     buttonAction[ButtonCountMax];                           //!< ボタンが押されたときの挙動です。
        char16_t    buttonMessage[ButtonCountMax][ButtonMessageLengthMax];  //!< ボタン上のメッセージ（UTF-16）です。
        char16_t    message[MessageLengthMax];                              //!< メッセージ（UTF-16）です。
    };

    static const int ServerCodeLengthMax = 9;                               //!< エラービューアアプレットのサーバーコード（半角英数字）の最大長です。NULL終端の分を含めています。サーバーコード自体は最大8桁の英数字です。

    /**
    * @brief        エラービューアアプレットに渡された入力パラメータを元にクラスを初期化します。
    * @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ。
    * @pre
    *               - GetErrorType(startupParam) の返り値が SystemData, SystemError, RecordedSystemData, RecordedSystemError, RecordedApplicationError のいずれか
    */
    explicit ErrorInfo(const void* startupParam) NN_NOEXCEPT;

    /**
    * @brief        エラービューアアプレットに渡された入力パラメータとアプリケーションエラーコードのカテゴリを元にクラスを初期化します。
    * @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ。
    * @param[in]    category アプリケーションエラーコードのカテゴリ。起動中のアプリケーションのアプリケーション管理データを ns の API を使用して取得し、渡してください。
    * @pre
    *               - GetErrorType(startupParam) == ApplicationError
    */
    ErrorInfo(const void* startupParam, const ns::ApplicationErrorCodeCategory& category) NN_NOEXCEPT;

    /**
    * @brief        与えられた ErrorCode に対応するシステムデータを読み込みクラスを初期化します。
    * @param[in]    errorCode エラーコード。
    * @pre
    *               - errorCode.IsValie() == true
    */
    explicit ErrorInfo(const ErrorCode& errorCode) NN_NOEXCEPT;

    /**
    * @brief        与えられた Result に対応するシステムデータを読み込みクラスを初期化します。
    * @param[in]    result リザルト。
    * @pre
    *               - result.IsFailure() == true
    */
    explicit ErrorInfo(const nn::Result& result) NN_NOEXCEPT;

    /**
    * @brief        与えられた ErrorCode とメッセージをもとにクラスを初期化します。
    * @param[in]    errorCode エラーコード。
    * @param[in]    dialogMessage ダイアログ表示のメッセージ（UTF-8）。
    * @param[in]    fullScreenMessage 全画面表示のメッセージ（UTF-8）。
    * @param[in]    languageCode ボタン等の UI 部品に使用する言語。
    * @pre
    *               - errorCode.IsValie() == true
    *               - dialogMessage を UTF-16 に変換した際のバッファサイズが ErrorViewerMessageBufferSizeMax バイト以下。
    *               - fullScreenMessage を UTF-16 に変換した際のバッファサイズが ErrorViewerMessageBufferSizeMax バイト以下。
    */
    ErrorInfo(const ErrorCode& errorCode, const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT;

    /**
    * @brief        アプリケーションエラーコードとメッセージを指定してクラスを初期化します。
    * @param[in]    category アプリケーションエラーコードのカテゴリ。
    * @param[in]    number アプリケーションエラーコードの番号。
    * @param[in]    dialogMessage ダイアログ表示のメッセージ（UTF-8）。
    * @param[in]    fullScreenMessage 全画面表示のメッセージ（UTF-8）。
    * @param[in]    languageCode ボタン等の UI 部品に使用する言語。
    * @pre
    *               - dialogMessage を UTF-16 に変換した際のバッファサイズが ErrorViewerMessageBufferSizeMax バイト以下。
    *               - fullScreenMessage を UTF-16 に変換した際のバッファサイズが ErrorViewerMessageBufferSizeMax バイト以下。
    */
    ErrorInfo(const ns::ApplicationErrorCodeCategory& category, ApplicationErrorCodeNumber number,
        const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT;

    /**
    * @brief        有効なエラー情報を持っているかどうかを返します。
    */
    bool IsValid() const NN_NOEXCEPT;

    /**
    * @brief        エラーコードのテキスト表現を取得します。
    * @param[out]   codeString エラーコードのテキスト表現（UTF-16）。
    * @param[in]    maxLength codeString の長さ。
    * @pre
    *               - IsValid() == true
    *               - codeString != null
    *               - maxLength > 0
    */
    void GetErrorCodeString(char16_t* codeString, size_t maxLength) const NN_NOEXCEPT;

    /**
    * @brief        サーバー障害が原因の可能性があるエラーかどうかを返します。
    * @return       サーバー障害が原因の可能性のあるエラーであれば true。
    * @pre
    *               - IsValid() == true
    */
    bool IsServerError() const NN_NOEXCEPT;

    /**
    * @brief        サーバー障害が原因の可能性があるエラーについて、障害情報を取得するためのサーバーコードを取得します。
    * @return       サーバーコードを表す文字列。null 終端された最大 8 文字の半角英数字。
    * @pre
    *               - IsValid() == true
    *               - IsServerFailure() == true
    */
    const char* GetServerCode() const NN_NOEXCEPT;

    /**
    * @brief        ダイアログ表示をサポートしたエラーかどうかを返します。
    * @return       ダイアログ表示をサポートしていれば true。
    * @pre
    *               - IsValid() == true
    */
    bool IsDialogViewSupported() const NN_NOEXCEPT;

    /**
    * @brief        全画面表示をサポートしたエラーかどうかを返します。
    * @return       全画面表示をサポートしていれば true。
    * @pre
    *               - IsValid() == true
    */
    bool IsFullScreenViewSupported() const NN_NOEXCEPT;

    /**
    * @brief        ダイアログ表示に必要な情報を返します。
    * @pre
    *               - IsValid() == true
    *               - IsDialogViewSupported() == true
    */
    const UiSettings& GetDialogViewUiSettings() const NN_NOEXCEPT;

    /**
    * @brief        全画面表示に必要な情報を返します。
    * @pre
    *               - IsValid() == true
    *               - IsFullScreenViewSupported() == true
    */
    const UiSettings& GetFullScreenViewUiSettings() const NN_NOEXCEPT;

    /**
    * @brief        エラー情報に付随したフラグ集合の値を返します。
    * @return       エラー情報に付随したフラグ集合の値。
    */
    uint32_t GetAttributeFlagSetValue() const NN_NOEXCEPT;

    /**
    * @brief        エラー情報の元となった Result を返します。
    * @return       エラー情報の元となった Result。
    * @details      メッセージ指定のエラーなど Result から作成された ErrorInfo でない場合は ResultSuccess を返します。
    */
    nn::Result GetResult() const NN_NOEXCEPT;

private:
    ErrorType                           m_ErrorType;
    ErrorCode                           m_ErrorCode;
    ns::ApplicationErrorCodeCategory    m_ApplicationErrorCodeCategory;
    ApplicationErrorCodeNumber          m_ApplicationErrorCodeNumber;
    UiSettings                          m_DialogViewUiSettings;
    UiSettings                          m_FullScreenViewUiSettings;
    bool                                m_IsServerError;
    char                                m_ServerCode[ServerCodeLengthMax];
    uint32_t                            m_AttributeFlagSetValue;
    nn::Result                          m_Result;

    void InitializeWithMessages(const char* dialogMessage, const char* fullScreenMessage, const settings::LanguageCode& languageCode) NN_NOEXCEPT;
    void InitializeWithMessageForRecordedError(const char* message) NN_NOEXCEPT;
    void ReadSystemData(const ErrorCode& errorCode) NN_NOEXCEPT;
};

}}
