﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief エラービューアアプレット向けAPI
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time/time_PosixTime.h>
#include <nn/err/err_Api.h>
#include <nn/err/err_ErrorInfo.h>
#include <nn/err/err_SystemTypes.h>

namespace nn { namespace err {

//! @name エラービューアアプレット向けの API
//! @{

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、エラーの種類を取得します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       エラービューアアプレット表示関数から渡されたエラーの種類
*/
ErrorType GetErrorType(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、メッセージ指定のエラーにおいて与えられた言語を取得します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       言語
* @pre
*               - GetErrorType(startupParam) == SystemError || GetErrorType(startupParam) == ApplicationError
*/
nn::settings::LanguageCode GetMessageLanguageCode(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、EULA 表示の要求時に与えられたリージョン情報を取得します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       リージョン情報
* @pre
*               - GetErrorType(startupParam) == Eula || GetErrorType(startupParam) == SystemUpdateEula
*/
nn::settings::system::RegionCode GetEulaRegionCode(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        本体更新データの EULA 表示用にエラービューアアプレットに渡された追加の入力パラメータを解釈し、EULA のデータを取得します。
* @param[out]   pOutActualSize EULA のデータの実際のサイズ
* @param[out]   outBuffer EULA のデータ
* @param[in]    outBufferSize outBuffer のサイズ
* @param[in]    eulaData 本体更新データの EULA 表示用にエラービューアアプレットに渡された入力パラメータ
* @param[in]    language 取得する EULA の言語
* @return       指定した言語のデータが見つかった場合は true、見つからなかった場合は false。
* @details      本体更新データの EULA 表示の要求の際は、エラービューアアプレットの共通のパラメータに加えて EULA のデータが呼び出し元から Push されます。
*               Pop した EULA のデータを eulaData に与えて本関数を呼び出すことで、指定した言語の EULA データを取得することができます。
*/
bool GetEulaData(size_t* pOutActualSize, void* outBuffer, size_t outBufferSize, const void* eulaData, settings::Language language) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、ペアレンタルコントロールにより制限された機能に対応した Result を取得します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       ペアレンタルコントロールにより制限された機能に対応した Result
* @pre
*               - GetErrorType(startupParam) == ParentalControl
* @details      返り値に対して @ref nn::pctl モジュールで定義された Result の Includes 関数を呼ぶことで、制限された機能を調べる事ができます。
*/
nn::Result GetParentalControlResultRestricted(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、記録されたエラーの表示かどうかを判定します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       エラー履歴からの表示の場合は true
*/
bool IsErrorRecordDisplayRequested(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、エラー履歴からの表示の場合にエラーの発生日時を取得します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       エラーの発生日時（協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数）
* @pre
*               - IsErrorRecordDisplayRequested(startupParam) == true
*/
nn::time::PosixTime GetTimeOfOccurrence(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットに渡された入力パラメータを解釈し、エラービューアアプレットから他のアプレット（アプリケーション含む）へのジャンプが有効かどうかを判定します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @return       ジャンプが有効な場合は true
* @details      この関数が false の時に他のアプレットへのジャンプが必要なボタンが押された場合、
*               エラービューアアプレットはジャンプを実行せず、SetErrorViewerResult() でジャンプ先の情報を設定して終了してください。
*               ジャンプはエラービューアアプレット終了後に呼び出し元のアプレットから行います。
*/
bool IsJumpEnabled(const void* startupParam) NN_NOEXCEPT;

/**
* @brief        エラービューアアプレットの終了パラメータを設定します。
* @param[in]    destination ジャンプ先
* @details      IsJumpEnabled() == false の場合、
*               エラービューアアプレットはジャンプを実行せず、この関数でジャンプ先の情報を終了パラメータとして設定して終了して下さい。
*               ジャンプはエラービューアアプレット終了後に呼び出し元のアプレットから行います。
*/
void SetErrorViewerResult(ErrorViewerJumpDestination destination) NN_NOEXCEPT;

/**
* @brief        エラーをシステムに保存します。
* @param[in]    startupParam エラービューアアプレットに渡された入力パラメータ
* @param[in]    workBuffer エラーレポートの作成に使用するワークバッファ
* @param[in]    workBufferSize workBuffer のサイズ
* @pre
*               - GetErrorType(startupParam) == SystemData || GetErrorType(startupParam) == SystemError || GetErrorType(startupParam) == ApplicationError
*               - workBufferSize >= RequiredBufferSizeForRecordError
*               - ns::InitializeForApplicationManager() を呼び出し済み
* @details      保存したエラーは本体設定のエラー履歴での閲覧対象になります。
*/
void RecordError(const void* startupParam, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
* @brief        エラー情報に付随したフラグに応じた処理を行います。
* @param[in]    errorInfo エラー情報。
* @pre
*               - nn::ns::InitializeForApplicationManager() を呼び出し済み
* @details      ErrorType::SystemData のエラー情報を表示した際に呼び出してください。
*               エラーコードDBでエラー情報に設定されたフラグに応じた処理を行います。
*/
void ProcessErrorInfoAttributes(const ErrorInfo& errorInfo) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------

}}
