﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief クライアントから消費型サービスアイテムを利用するためのAPIの定義
 */

#pragma once
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/err.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace ec {

/**
    @brief  消費型サービスアイテムに関する非同期要求の基底クラス

    @details
        本クラスは、消費型サービスアイテムの非同期要求について、共通処理を定義した基底クラスです。
        要求内容ごとに変わる処理は、各派生クラスに定義されます。基底クラス単体ではインスタンスを生成できません。

        基底クラスでは、以下の機能を提供します。

        - Initialize()      非同期要求の初期化
        - Cancel()          非同期要求のキャンセル
        - Wait()            非同期要求の完了待ち
        - TryWait()         非同期要求の完了待ち確認
        - GetEvent()        非同期要求完了通知を受けるイベントの取得
        - GetErrorCode()    非同期要求結果のエラーコード取得

        派生クラスでは、以下の機能を提供します。

        - Begin()           非同期要求の開始
        - End()             非同期要求の終了
*/
class ConsumableServiceItemAsyncRequestBase
{
    NN_DISALLOW_COPY(ConsumableServiceItemAsyncRequestBase);
    NN_DISALLOW_MOVE(ConsumableServiceItemAsyncRequestBase);

public:

    /**
        @brief 非同期要求を初期化します。

        @param[in]  pAccessor       OwnedConsumableServiceItem で初期化したアクセッサ
        @param[in]  user            権利情報を参照するユーザ
        @param[in]  workMemory      ワークメモリ
        @param[in]  workMemorySize  ワークメモリのサイズ

        @pre
            - pAccessor != nullptr
            - static_cast<bool>(user) == true
            - workMemory != nullptr
            - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

        @details
            pAccessor で入力されるアクセッサは、nn::ec::ShopServiceAccessor のリファレンスを参照して適切に初期化してください。
            初期化が適切に行われていない場合、非同期要求に関する操作で、ShopServiceAccessor のエラーが返ることがあります。
    */
    void Initialize(ShopServiceAccessor* pAccessor, const nn::account::Uid& user, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;

    /**
        @brief 非同期要求をキャンセルします。
    */
    void Cancel() NN_NOEXCEPT;

    /**
        @brief 非同期要求の完了を待ちます。
    */
    void Wait() NN_NOEXCEPT;

    /**
        @brief 非同期要求の完了待ち確認を行います。

        @return 非同期要求の完了状態を返します。
    */
    bool TryWait() NN_NOEXCEPT;

    /**
        @brief 非同期要求完了通知を受けるイベントオブジェクトの参照を取得します。

        @return 非同期要求完了通知を受けるイベントオブジェクトの参照を返します。
    */
    nn::os::SystemEvent& GetEvent() NN_NOEXCEPT;

    /**
        @brief 非同期要求結果のエラーコードを取得します。

        @return 非同期要求結果のエラーコードを返します。
    */
    nn::err::ErrorCode GetErrorCode() const NN_NOEXCEPT;

protected:

    /**
        @brief 基底クラスのコンストラクタ

        @details インスタンス生成は禁止されています。
    */
    ConsumableServiceItemAsyncRequestBase() NN_NOEXCEPT
        : m_IsInitialized(false)
    {}

    bool IsRequestAvailable() const NN_NOEXCEPT;
    Result GetSize(size_t* pOutValue) NN_NOEXCEPT;
    Result Get(void* pOutReceivedBuffer, size_t bufferCapacity) NN_NOEXCEPT;
    ShopServiceAccessor* GetAccessor() NN_NOEXCEPT;
    nn::account::Uid GetUser() NN_NOEXCEPT;
    void* GetWorkMemory() NN_NOEXCEPT;
    size_t GetWorkMemorySize() NN_NOEXCEPT;
    ShopServiceAccessor::AsyncResponse& GetAsyncResponse() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;

private:
    ShopServiceAccessor * m_pAccessor;
    nn::account::Uid m_User;
    void* m_pWorkMemory;
    size_t m_WorkMemorySize;
    ShopServiceAccessor::AsyncResponse m_Response;
    bool m_IsInitialized;
};

/**
    @brief  サーバーに所有している未消費の権利情報を取得する非同期要求クラス
*/
class AsyncGetConsumableRightDataRequest : public ConsumableServiceItemAsyncRequestBase
{
    NN_DISALLOW_COPY(AsyncGetConsumableRightDataRequest);
    NN_DISALLOW_MOVE(AsyncGetConsumableRightDataRequest);

public:

    /**
        @brief コンストラクタ
    */
    AsyncGetConsumableRightDataRequest() NN_NOEXCEPT
        : m_IsRequestBegin(false)
    {}

    /**
        @brief 未消費の権利情報の取得を開始します。

        @param[in]  page                ページインデックス
        @param[in]  perPage             1 ページあたりの取得件数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessOverRequest                  非同期要求に必要なサービスリソースを確保できませんでした。
                                                                                未破棄の非同期要求があれば破棄した後に再度要求してください。
                                                                                もしくは、ShopServiceAccessor の同時利用数を減らしてください。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInsufficientWorkMemory                        ConsumableServiceItemAsyncRequestBase のワークメモリが不足しています。
                                                                                初期化時に nn::ec::RequiredWorkMemorySizeForConsumableServiceItem byte 以上のワークメモリを渡してください。
        @retval     nn::ec::ResultInternetRequestNotAccepted                    インターネット接続要求が提出、及び受理されていません。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - nn::nifm::IsNetworkAvailable() == true
            - page >= 0
            - perPage > 0
            - perPage <= ConsumableServiceItemRightDataCountMax

        @details
            初期化時に指定したユーザが、サーバーに所有している権利情報を取得します。権利情報の一覧は、購入時刻の降順で表示されます。
            本 API では、1 ページあたりの取得する権利数と、取得するページインデックスを指定できます。

            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            本 API では、サーバーへの要求を行います。サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
            制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
    */
    Result Begin(int page, int perPage) NN_NOEXCEPT;

    /**
        @brief 未消費な権利情報の取得完了を待ちます。

        @param[out] pOutRightDataCount  取得した権利情報の数
        @param[out] outRightDataArray   取得した権利情報の格納先
        @param[in]  maxRightDataCount   取得する権利情報の最大数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessCanceled                     呼び出し元からの要求に従い処理がキャンセルされました。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInsufficientWorkMemory                        ConsumableServiceItemAsyncRequestBase のワークメモリが不足しています。
                                                                                初期化時に nn::ec::RequiredWorkMemorySizeForConsumableServiceItem byte 以上のワークメモリを渡してください。
        @retval     nn::ec::ResultShopServiceAccessRequestTimeout               サーバーとの通信要求がタイムアウトしました。
        @retval     nn::account::ResultNetworkServiceAccountUnavailable         指定されたユーザアカウントのネットワークサービスアカウントが利用できません。
                                                                                ACCOUNT ライブラリのエラーハンドリングリファレンスを参考にネットワークサービスアカウントを利用可能状態にしてください。
        @retval     nn::ec::ResultShowErrorCodeRequired                         接続先ショップサービスサーバーからエラービューア表示が必要なレスポンスエラーコードを受信しました。
        @retval     nn::ec::ResultConsumableServiceItemInvalidServerRightStatus サーバー上で不正な状態の権利情報を検出しました。
        @retval     nn::ec::ResultConsumableServiceItemInsufficientBuffer       出力バッファが不足しています。
                                                                                outRightDataArray には、Begin 実行時に指定した perPage 分以上のバッファを用意してください。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - Begin() が実行済み
            - nn::nifm::IsNetworkAvailable() == true
            - pOutRightDataCount != nullptr
            - outRightDataArray != nullptr
            - maxRightDataCount >= Begin() 実行時に指定した perPage
            - maxRightDataCount <= ConsumableServiceItemRightDataCountMax

        @details
            非同期処理が完了していなければ、API 内部で完了を待機します。

            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            レスポンスに含まれている権利数に関わらず、一度に maxRightDataCount 個まで権利情報を取得できます。
            outRightDataArray には、Begin 実行時に指定した perPage 分以上のバッファを用意してください。
    */
    Result End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT;

private:
    int m_Page;
    int m_PerPage;
    bool m_IsRequestBegin;
};

/**
    @brief  サーバーに所有している権利情報を消費する非同期要求クラス
*/
class AsyncConsumeRightDataRequest : public ConsumableServiceItemAsyncRequestBase
{
    NN_DISALLOW_COPY(AsyncConsumeRightDataRequest);
    NN_DISALLOW_MOVE(AsyncConsumeRightDataRequest);

public:

    /**
        @brief コンストラクタ
    */
    AsyncConsumeRightDataRequest() NN_NOEXCEPT
        : m_IsRequestBegin(false)
    {}

    /**
        @brief 未消費な権利情報の消費を開始します。

        @param[in]  rightDataArray      消費する権利情報
        @param[in]  rightDataCount      消費する権利情報の数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessOverRequest                  非同期要求に必要なサービスリソースを確保できませんでした。
                                                                                未破棄の非同期要求があれば破棄した後に再度要求してください。
                                                                                もしくは、ShopServiceAccessor の同時利用数を減らしてください。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInsufficientWorkMemory                        ConsumableServiceItemAsyncRequestBase のワークメモリが不足しています。
                                                                                初期化時に nn::ec::RequiredWorkMemorySizeForConsumableServiceItem byte 以上のワークメモリを渡してください。
        @retval     nn::ec::ResultInternetRequestNotAccepted                    インターネット接続要求が提出、及び受理されていません。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - nn::nifm::IsNetworkAvailable() == true
            - rightDataArray != nullptr
            - rightDataArray[i].IsValid == true
            - rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax

        @details
            初期化時に指定したユーザが、サーバーに所有している権利情報を消費します。
            rightDataArray に入力した権利情報のうち、IsServiceProvidableOnLocal() == true を満たすものは API 内で権利消費がスキップされます。
            消費した権利は以後、AsyncGetConsumableRightDataRequest では参照できなくなります。

            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            本 API では、サーバーへの要求を行います。サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
            制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
    */
    Result Begin(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT;

    /**
        @brief 権利情報の消費完了を待ちます

        @param[out] pOutRightDataCount  消費できた権利情報の数
        @param[out] outRightDataArray   消費できた権利情報の格納先
        @param[in]  maxRightDataCount   取得する権利情報の最大数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessCanceled                     呼び出し元からの要求に従い処理がキャンセルされました。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInsufficientWorkMemory                        ConsumableServiceItemAsyncRequestBase のワークメモリが不足しています。
                                                                                初期化時に nn::ec::RequiredWorkMemorySizeForConsumableServiceItem byte 以上のワークメモリを渡してください。
        @retval     nn::ec::ResultShopServiceAccessRequestTimeout               サーバーとの通信要求がタイムアウトしました。
        @retval     nn::account::ResultNetworkServiceAccountUnavailable         指定されたユーザアカウントのネットワークサービスアカウントが利用できません。
                                                                                ACCOUNT ライブラリのエラーハンドリングリファレンスを参考にネットワークサービスアカウントを利用可能状態にしてください。
        @retval     nn::ec::ResultShowErrorCodeRequired                         接続先ショップサービスサーバーからエラービューア表示が必要なレスポンスエラーコードを受信しました。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - Begin() が実行済み
            - nn::nifm::IsNetworkAvailable() == true
            - pOutRightDataCount != nullptr
            - outRightDataArray != nullptr
            - maxRightDataCount >= Begin() 実行時に指定した rightDataCount
            - maxRightDataCount > 0

        @details
            非同期処理が完了していなければ、API 内部で完了を待機します。

            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            消費できた権利は、以後、IsServiceProvidableOnLocal() == true を満たします。
            outRightDataArray には、Begin 実行時に指定した rightDataCount 分以上のバッファを用意してください。
    */
    Result End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT;

private:
    const ConsumableServiceItemRightData* m_pRightData;
    int m_RightDataCount;
    bool m_IsRequestBegin;
};

/**
    @brief  提供可能な権利情報を取得する非同期要求クラス
*/
class AsyncGetProvidableRightDataRequest : public ConsumableServiceItemAsyncRequestBase
{
    NN_DISALLOW_COPY(AsyncGetProvidableRightDataRequest);
    NN_DISALLOW_MOVE(AsyncGetProvidableRightDataRequest);

public:

    /**
        @brief コンストラクタ
    */
    AsyncGetProvidableRightDataRequest() NN_NOEXCEPT
        : m_IsRequestBegin(false)
    {}

    /**
        @brief 提供可能な権利情報の取得を開始します。

        @param[in]  rightDataArray      提供可否判定する権利情報
        @param[in]  rightDataCount      提供可否判定する権利情報の数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessOverRequest                  非同期要求に必要なサービスリソースを確保できませんでした。
                                                                                未破棄の非同期要求があれば破棄した後に再度要求してください。
                                                                                もしくは、ShopServiceAccessor の同時利用数を減らしてください。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInternetRequestNotAccepted                    インターネット接続要求が提出、及び受理されていません。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - nn::nifm::IsNetworkAvailable() == true
            - rightDataArray != nullptr
            - rightDataArray[i].IsValid == true
            - rightDataCount > 0
            - rightDataCount <= ConsumableServiceItemRightDataCountMax

        @details
            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            本 API では、サーバーへの要求を行います。サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
            制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
    */
    Result Begin(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT;

    /**
        @brief 提供可能な権利情報の取得完了を待ちます

        @param[out] pOutRightDataCount  提供可能な権利情報の数
        @param[out] outRightDataArray   提供可能な権利情報の格納先
        @param[in]  maxRightDataCount   取得する権利情報の最大数

        @retval     nn::ResultSuccess                                           処理が成功しました。
        @retval     nn::ec::ResultShopServiceAccessCanceled                     呼び出し元からの要求に従い処理がキャンセルされました。
        @retval     nn::ec::ResultShopServiceAccessInsufficientWorkMemory       ShopServiceAccessor のワークメモリが不足しています。
        @retval     nn::ec::ResultInsufficientWorkMemory                        ConsumableServiceItemAsyncRequestBase のワークメモリが不足しています。
                                                                                初期化時に nn::ec::RequiredWorkMemorySizeForConsumableServiceItem byte 以上のワークメモリを渡してください。
        @retval     nn::ec::ResultShopServiceAccessRequestTimeout               サーバーとの通信要求がタイムアウトしました。
        @retval     nn::account::ResultNetworkServiceAccountUnavailable         指定されたユーザアカウントのネットワークサービスアカウントが利用できません。
                                                                                ACCOUNT ライブラリのエラーハンドリングリファレンスを参考にネットワークサービスアカウントを利用可能状態にしてください。
        @retval     nn::ec::ResultShowErrorCodeRequired                         接続先ショップサービスサーバーからエラービューア表示が必要なレスポンスエラーコードを受信しました。
        @retval     nn::ec::ResultConsumableServiceItemInvalidServerRightStatus サーバー上で不正な状態の権利情報を検出しました。
        @retval     nn::ec::ResultConsumableServiceItemInsufficientBuffer       出力バッファが不足しています。
                                                                                outRightDataArray には、Begin 実行時に指定した rightDataCount 分以上のバッファを用意してください。

        @pre
            - Initialize() 実行済み
            - Initialize() 実行時に指定したアクセッサが初期化済み
            - Begin() が実行済み
            - nn::nifm::IsNetworkAvailable() == true
            - pOutRightDataCount != nullptr
            - outRightDataArray != nullptr
            - maxRightDataCount >= Begin() 実行時に指定した rightDataCount
            - maxRightDataCount > 0

        @details
            非同期処理が完了していなければ、API 内部で完了を待機します。

            本 API では、nn::ec::ShopServiceAccessor::AsyncResponse のエラーが返ることがあります。
            そのエラーハンドリングについては、ShopServiceAccessor::AsyncResponse のリファレンスを参照してください。

            レスポンスに含まれている権利数に関わらず、一度に maxRightDataCount 個まで権利情報を取得できます。
            全ての権利情報を取得する場合、Begin 実行時に指定した rightDataCount 分以上のバッファを用意してください。

            提供可能と判断された権利は、以後、IsServiceProvidableOnLocal() == true を満たします。
            outRightDataArray には、Begin 実行時に指定した rightDataCount 分以上のバッファを用意してください。
    */
    Result End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT;

private:
    const ConsumableServiceItemRightData* m_pRightData;
    int m_RightDataCount;
    bool m_IsRequestBegin;
};

/**
    @brief ConsumableServiceItemRightData をシリアライズするのに必要なバッファサイズを取得します。

    @param[in]  rightDataArray              権利情報
    @param[in]  rightDataCount              権利情報の数

    @pre
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0

    @return     必要なバッファサイズを返します
*/
size_t GetRequiredBufferSizeToSerializeConsumableServiceItemRightData(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT;

/**
    @brief セーブデータで保存する形式に、ConsumableServiceItemRightData をシリアライズします

    @param[out] buffer                      シリアライズしたデータの格納先
    @param[in]  bufferSize                  buffer の領域サイズ
    @param[in]  rightDataArray              権利情報
    @param[in]  rightDataCount              権利情報の数

    @pre
        - buffer != nullptr
        - bufferSize >= GetRequiredBufferSizeToSerializeConsumableServiceItemRightData(rightDataArray, rightDataCount)
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0

    @return     シリアライズしたデータサイズを返します
*/
size_t SerializeConsumableServiceItemRightData(void* buffer, size_t bufferSize, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT;

/**
    @brief シリアライズした ConsumableServiceItemRightData について、デシリアライズするのに必要な情報を取得します。

    @param[out] pOutInfo                    デシリアライズするのに必要な情報
    @param[in]  buffer                      データ
    @param[in]  bufferSize                  データサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
    @endretresult
    @retresult
        @handleresult{
            nn::ec::ResultConsumableServiceItemInvalidSaveData(),
            入力された buffer, bufferSize ではデシリアライズできません。エラービュワーに表示してください。
        }
    @endretresult

    @pre
        - pOutInfo != nullptr
        - buffer != nullptr
        - bufferSize > 0
*/
Result GetSerializedConsumableServiceItemRightDataInfo(SerializedConsumableServiceItemRightDataInfo *pOutInfo, const void* buffer, const size_t bufferSize) NN_NOEXCEPT;

/**
    @brief シリアライズした ConsumableServiceItemRightData をデシリアライズします

    @param[out] outRightDataArray           デシリアライズした権利情報の格納先
    @param[in]  maxRightDataCount           権利情報の最大数
    @param[in]  buffer                      データ
    @param[in]  bufferSize                  データサイズ

    @pre
        - outRightDataArray != nullptr
        - maxRightDataCount > 0
        - buffer != nullptr
        - bufferSize > 0
        - GetSerializedConsumableServiceItemRightDataInfo(pOutInfo, buffer, bufferSize).IsSuccess() == true
          maxRightDataCount >= pOutInfo->rightDataCount

    @return     デシリアライズした権利情報の数を返します
*/
int DeserializeConsumableServiceItemRightData(ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const void* buffer, size_t bufferSize) NN_NOEXCEPT;

/* --------------------------------------------------------------------------------------------
    実装
*/
#if !defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

inline void ConsumableServiceItemAsyncRequestBase::Initialize(ShopServiceAccessor* pAccessor, const nn::account::Uid& user, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(static_cast<bool>(user) == true);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);
    NN_SDK_REQUIRES_NOT_NULL(workMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);

    m_pAccessor = pAccessor;
    m_User = user;
    m_pWorkMemory = workMemory;
    m_WorkMemorySize = workMemorySize;
    m_IsInitialized = true;
}

inline void ConsumableServiceItemAsyncRequestBase::Cancel() NN_NOEXCEPT
{
    m_Response.Cancel();
}

inline void ConsumableServiceItemAsyncRequestBase::Wait() NN_NOEXCEPT
{
    m_Response.Wait();
}

inline bool ConsumableServiceItemAsyncRequestBase::TryWait() NN_NOEXCEPT
{
    return m_Response.TryWait();
}

inline nn::os::SystemEvent& ConsumableServiceItemAsyncRequestBase::GetEvent() NN_NOEXCEPT
{
    return m_Response.GetEvent();
}

inline nn::err::ErrorCode ConsumableServiceItemAsyncRequestBase::GetErrorCode() const NN_NOEXCEPT
{
    return m_Response.GetErrorCode();
}

inline Result ConsumableServiceItemAsyncRequestBase::GetSize(size_t* pOutValue) NN_NOEXCEPT
{
    return m_Response.GetSize(pOutValue);
}

inline Result ConsumableServiceItemAsyncRequestBase::Get(void* pOutReceivedBuffer, size_t bufferCapacity) NN_NOEXCEPT
{
    return m_Response.Get(pOutReceivedBuffer, bufferCapacity);
}

inline ShopServiceAccessor* ConsumableServiceItemAsyncRequestBase::GetAccessor() NN_NOEXCEPT
{
    return m_pAccessor;
}

inline nn::account::Uid ConsumableServiceItemAsyncRequestBase::GetUser() NN_NOEXCEPT
{
    return m_User;
}

inline void* ConsumableServiceItemAsyncRequestBase::GetWorkMemory() NN_NOEXCEPT
{
    return m_pWorkMemory;
}

inline size_t ConsumableServiceItemAsyncRequestBase::GetWorkMemorySize() NN_NOEXCEPT
{
    return m_WorkMemorySize;
}

inline ShopServiceAccessor::AsyncResponse& ConsumableServiceItemAsyncRequestBase::GetAsyncResponse() NN_NOEXCEPT
{
    return m_Response;
}

inline bool ConsumableServiceItemAsyncRequestBase::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

#endif

}}
