﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a Header class provided
 * by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a Header class represents the DNS header section of a
 * well-formed DNS message.
 *
 * @details
 * The header is the most basic section of a DNS message, it is always
 * included. It contains flags that provide context for the message
 * and a transaction identifier that associates a  request and
 * response message. The header also contains a count of questions and
 * types of resource records.
 */
class Header
{
private:
    /**
     * @brief A pointer to the nn::dns::parser::Message that contains this object.
     */
    const Message* m_pMessage;

    /**
     * @brief The block of memory that refers to the buffer
     * that created the @a Header with @ref Header::FromBuffer().
     */
    MemoryBlock m_Range;

    /**
     * @brief Indicates the object size or if it is uninitialized.
     */
    size_t m_DirtySize;

    /**
     * @brief The 16-bit DNS transaction ID.
     *
     * @details
     * The transaction ID is supposed to be unique. However 16 bits of
     * entropy may not be sufficient to use in a complex application.
     * In that case the transaction ID could be combined other network
     * data to produce an acceptable value.
     */
    uint16_t m_TransactionId;

    /**
     * @brief The flags value with bits that indicate how the
     * message ought to be interpreted.
     *
     * @details
     *
     * Constants for accessing these bits (Shift / Mask) are provided
     * in the HeaderConstant enum.
     *
     * Request bits:
     * -  0-3: Unused.
     * -    4: Indicates if non-authenticated data is acceptable.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableShift  and nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableMask.
     * -    5: Unused.
     * -    6: The reserved "Zero" bit.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::ZeroShift and nn::dns::parser::HeaderFlagsConstant::ZeroMask.
     * -    7: Unused.
     * -    8: Indicates recursion is desired.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift and nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredMask.
     * -    9: Indicates if the message was truncated.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedShift and nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedMask.
     * -   10: Unused.
     * -11-14: The operation code.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::OperationCodeShift and nn::dns::parser::HeaderFlagsConstant::OperationCodeMask.
     * -   15: If set to 0 then it indicates the message is a request.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsResponseShift and nn::dns::parser::HeaderFlagsConstant::IsResponseMask.
     *
     * Response bits:
     * -  0-3: The response code from-server code.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::ResponseCodeShift & nn::dns::parser::HeaderFlagsConstant::ResponseCodeMask.
     * -    4: Indicates if non-authenticated data is acceptable.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableShift  and nn::dns::parser::HeaderFlagsConstant::IsNonAuthenticatedAcceptableMask.
     * -    5: Indicates that the message was authenticated.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsResponseAuthenticatedShift and nn::dns::parser::HeaderFlagsConstant::IsResponseAuthenticatedMask.
     * -    6: The reserved "Zero" bit.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::ZeroShift and nn::dns::parser::HeaderFlagsConstant::ZeroMask.
     * -    7: Indicates if recursion was available.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsRecursionAvailableShift and nn::dns::parser::HeaderFlagsConstant::IsRecursionAvailableMask.
     * -    8: Indicates recursion was desired.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift and nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredMask..
     * -    9: Indicates if the message was truncated.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedShift and nn::dns::parser::HeaderFlagsConstant::IsMessageTruncatedMask.
     * -   10: Indicates if the server is authoritative.
     * -11-14: The operation code.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::OperationCodeShift and nn::dns::parser::HeaderFlagsConstant::OperationCodeMask.
     * -   15: If set to 1 then it indicates the message is a response.
     *         Accessed with nn::dns::parser::HeaderFlagsConstant::IsResponseShift and nn::dns::parser::HeaderFlagsConstant::IsResponseMask.
     */
    HeaderFlagsConstant m_Flags;

    /**
     * @brief The number of question records in the DNS message.
     *
     * @details
     * - In The RFC it is called 'qdcount'.
     * - Multiple questions are very uncommon but they are supported.
     */
    uint16_t m_QuestionCount;

    /**
     * @brief The number of answer records in the DNS message.
     *
     * @details
     * In The RFC it is called 'ancount'.
     */
    uint16_t m_AnswerCount;

    /**
     * @brief The number of nameserver authority records in DNS message.
     *
     * @details
     * In The RFC it is called 'nscount'.
     */
    uint16_t m_AuthorityCount;

    /**
     * @brief The number of additional records. This section often
     * contains optional (OPT) records for the root domain (0) that
     * are used for EDNS0 / DNSSEC.
     *
     * @details
     * In The RFC it is called 'arcount'.
     */
    uint16_t m_AdditionalCount;

public:

    /**
     * @brief Constructor.
     */
    Header();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The @a Header on the right hand side of the
     * expression.
     */
    Header(const Header& rhs);

    /**
     * @brief Destructor.
     */
    ~Header();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The @a Header on the right hand side of the
     * expression.
     */
    Header& operator=(const Header& rhs);

    /**
     * @brief Returns true if two @a Header objects are equal.
     *
     * @param[in] rhs The @a Header on the right hand side of
     * the expression.
     *
     * @return Returns true if both objects are content equivalent.
     */
    bool operator==(const Header& rhs) const NN_NOEXCEPT;

    /**
     * @brief Getter for @a m_pMessage.
     *
     * @returns A reference to the nn::dns::parser::Message pointer.
     */
    const Message* & GetMessage() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_pTransactionId.
     *
     * @returns A reference to the transaction id header field.
     */
    uint16_t& GetTransactionId() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_Flags.
     *
     * @details For constants used to access this value refer to
     * nn::dns::parser::HeaderFlagsConstant.
     *
     * @returns A reference to the flags header field.
     */
    HeaderFlagsConstant& GetFlags() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_QuestionCount.
     *
     * @returns A reference to the question count header field.
     */
    uint16_t& GetQuestionCount() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_AnswerCount.
     *
     * @returns A reference to the answer count header field.
     */
    uint16_t& GetAnswerCount() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_AuthorityCount.
     *
     * @returns A reference to the authority count header field.
     */
    uint16_t& GetAuthorityCount() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_AdditionalCount.
     *
     * @returns A reference to the additional count header field.
     */
    uint16_t& GetAdditionalCount() NN_NOEXCEPT;

    /**
     * @brief Initializes the @a Header.
     *
     * @details
     * Initializes a @a Header with the provided nn::dns::parser::Message
     * zeroes out all fields, sets dirty flag to zero, and point the
     * @a Header at the provided nn::dns::parser::Message.
     *
     * @param[in] pMessage The provided @a Message.
     */
    void Initialize(const Message* pMessage) NN_NOEXCEPT;

    /**
     * @brief Get the size of a @a Header for serialization.
     *
     * @return Returns error or success:
     *         - -1 : An error occurred.
     *         - >0 : The size of the object.
     */
    ssize_t SizeOf() const NN_NOEXCEPT;

    /**
     * @brief Reads a @a Header from the contents of a buffer in the
     * standard RFC format.
     *
     * @details It is the reverse of @ref Header::ToBuffer().
     *
     * @param[in] pBuffer The buffer that contains @a Header bytes.
     *
     * @param[in] size The size in bytes left in the buffer.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include a null buffer pointer or parsing error.
     */
    ssize_t FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief Writes a @a Header to a buffer in the RFC standard
     * format.
     *
     * @details It is the reverse of @ref Header::FromBuffer().
     *
     * @param[out] pOutBuffer The buffer that receives @a Header bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @return Returns the number of bytes written to the buffer or
     * -1 on error.
     *
     * Error cases include null buffer pointer, a buffer of
     * insufficient size, or a parsing error.
     */
    ssize_t ToBuffer(uint8_t* const pOutBuffer, size_t size) const NN_NOEXCEPT;
};

}}}; // nn::dns::parser
