﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   USB Audio Class Specifications
 */

namespace nn {
namespace cduac {

/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.1
 */
enum AudioInterfaceClassCode
{
    Class_Audio                                 = 0x01,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.2
 */
enum AudioSubClassCode
{
    SubClass_Undefined                          = 0x00,
    SubClass_AudioControl                       = 0x01,
    SubClass_AudioStreaming                     = 0x02,
    SubClass_MidiStreaming                      = 0x03,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.3
 */
enum AudioProtocolCode
{
    Protocol_Undefined                          = 0x00,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.4
 */
enum DescriptorType
{
    DescriptorType_Undefined                    = 0x20,
    DescriptorType_Device                       = 0x21,
    DescriptorType_Configuration                = 0x22,
    DescriptorType_String                       = 0x23,
    DescriptorType_Interface                    = 0x24,
    DescriptorType_Endpoint                     = 0x25,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.5
 */
enum AudioControlDescriptorSubType
{

    AudioControl_UndefinedSubType               = 0x00,
    AudioControl_Header                         = 0x01,
    AudioControl_InputTerminal                  = 0x02,
    AudioControl_OutputTerminal                 = 0x03,
    AudioControl_MixerUnit                      = 0x04,
    AudioControl_SelectorUnit                   = 0x05,
    AudioControl_FeatureUnit                    = 0x06,
    AudioControl_ProcessingUnit                 = 0x07,
    AudioControl_ExtensionUnit                  = 0x08,

};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.6
 */
enum AudioStreamingInterfaceDescriptorSubType
{

    AudioStreaming_Undefined                    = 0x00,
    AudioStreaming_General                      = 0x01,
    AudioStreaming_FormatType                   = 0x02,
    AudioStreaming_FormatSpecific               = 0x03,

};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.7
 */
enum ProcessingUnitType
{
    ProcessingUnit_Undefined                    = 0x00,
    ProcessingUnit_UpDownMix                    = 0x01,
    ProcessingUnit_DolbyPrologic                = 0x02,
    ProcessingUnit_3dStereoExpander             = 0x03,
    ProcessingUnit_Reverberation                = 0x04,
    ProcessingUnit_Chorus                       = 0x05,
    ProcessingUnit_DynanmicReange               = 0x06,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.8
 */
enum AudioStreamingEndpointDescriptorSubType
{
    AudioStreaming_Endpoint_Undefined           = 0x00,
    AudioStreaming_Endpoint_General             = 0x01,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.9
 */
enum Request
{
    Request_Undefined                           = 0x00,
    Request_SetCur                              = 0x01,
    Request_GetCur                              = 0x81,
    Request_SetMin                              = 0x02,
    Request_GetMin                              = 0x82,
    Request_SetMax                              = 0x03,
    Request_GetMax                              = 0x83,
    Request_SetRes                              = 0x04,
    Request_GetRes                              = 0x84,
    Request_SetMem                              = 0x05,
    Request_GetMem                              = 0x85,
    Request_GetStat                             = 0xff,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.10.2
 */
enum AudioControl
{
    AudioControl_Undefined                      = 0x0000,
    AudioControl_Mute                           = 0x0001,
    AudioControl_Volume                         = 0x0002,
    AudioControl_Bass                           = 0x0004,
    AudioControl_Mid                            = 0x0008,
    AudioControl_Treble                         = 0x0010,
    AudioControl_GraphicEqualizer               = 0x0020,
    AudioControl_AutomaticGain                  = 0x0040,
    AudioControl_Delay                          = 0x0080,
    AudioControl_BassBoost                      = 0x0100,
    AudioControl_Loudness                       = 0x0200,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Appendix A.10.5
 */
enum Endpoint
{
    Endpoint_Undefined                          = 0x00,
    Endpoint_SamplingFrequencyControl           = 0x01,
    Endpoint_PitchControl                       = 0x02,
};


/**
 * @brief Referenced by structs below, used by parser to determine descriptor length, type, sub type
 */
struct AudioDescriptor
{

    uint8_t     bLength;
    uint8_t     bDescriptorType;
    uint8_t     bDescriptorSubtype;

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2
 */
struct AudioControlHeader
{

    AudioDescriptor audioDescriptor;
    uint16_t        bcdADC;
    uint16_t        wTotalLength;
    uint8_t         bInCollection;
    uint8_t         baInterfaceNr[1];

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.1
 */
struct AudioControlInputTerminal
{

    AudioDescriptor audioDescriptor;
    uint8_t         bTerminalId;
    uint16_t        wTermainalType;
    uint8_t         bAssocTerminal;
    uint8_t         bNrChannels;
    uint16_t        wChannelConfig;
    uint8_t         iChannelNames;
    uint8_t         iTermainal;

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.2
 */
struct AudioControlOutputTerminal
{

    AudioDescriptor audioDescriptor;
    uint8_t         bTerminalId;
    uint16_t        wTermainalType;
    uint8_t         bAssocTerminal;
    uint8_t         bSourceId;
    uint8_t         iTermainal;

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.4
 */
struct AudioControlSelectorUnit
{

    AudioDescriptor audioDescriptor;
    uint8_t         bUnitId;
    uint8_t         bNrInPins;
    uint8_t         baSourceId[1];

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.5
 */
struct AudioControlFeatureUnit
{

    AudioDescriptor audioDescriptor;
    uint8_t         bUnitId;
    uint8_t         bSourceId;
    uint8_t         bControlSize;
    uint8_t         bmaControls[1]; // array of contorls, [0] is master, [n] is channel n, number of channels determined by input terminal bNrChannels

    // after bmaControls[] is unit8_t iFeature, if > 0 is index to string table describing feature

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.3
 */
struct AudioControlMixerUnit
{

    AudioDescriptor audioDescriptor;
    uint8_t         bUnitId;
    uint8_t         bNrInPins;
    uint8_t         baSourceId[1]; // array of source ids. Length = bNrInPins

    //after baSourceId[]:
    //uint8_t bNrChannels;
    //uint16_t wChannelConfig;
    //uint8_t iChannelNames;
    //uint8_t bmControls[]; // Bitmap indicating which mixing controls are programmable
    //uint8_t iMixer;

} __attribute__((packed));

/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.3.2.5
 */
enum Control
{
    Control_Mute            = 1 << 0,
    Control_Volume          = 1 << 1,
    Control_Bass            = 1 << 2,
    Control_Mid             = 1 << 3,
    Control_Treble          = 1 << 4,
    Control_AutomaticGain   = 1 << 5,
    Control_Delay           = 1 << 6,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.5.2
 */
struct AudioStreamingGeneral
{

    AudioDescriptor audioDescriptor;
    uint8_t         bTerminalLink;
    uint8_t         bDelay;
    uint16_t        wFormatTag;

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 4.6.1.2
 */
struct AudioStreamingEndpointGeneral
{

    AudioDescriptor audioDescriptor;
    uint8_t         bmAttributes;
    uint8_t         bLockDelayUnits;
    uint16_t        wLockDelay;

} __attribute__((packed));


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 5.2.1
 */
enum RequestType
{
    RequestType_SetAudioControl                 = 0x21,
    RequestType_SetAudioStreaming               = 0x22,
    RequestType_GetAudioControl                 = 0xa1,
    RequestType_GetAudioStreaming               = 0xa2,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.1
 */
enum UsbTerminalType
{
    UsbTerminalType_Undefined                   = 0x0100,
    UsbTerminalType_Streaming                   = 0x0101,
    UsbTerminalType_Vendor                      = 0x01ff,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.2
 */
enum InputTerminalType
{
    InputTerminalType_Undefined                 = 0x0200,
    InputTerminalType_Microphone                = 0x0201,
    InputTerminalType_DesktopMicrophone         = 0x0202,
    InputTerminalType_PersonalMicrophone        = 0x0203,
    InputTerminalType_OmnidirectionalMicrophone = 0x0204,
    InputTerminalType_MicrophoneArray           = 0x0205,
    InputTerminalType_ProcessingMicArray        = 0x0206,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.3
 */
enum OutputTerminalType
{
    OutputTermainalType_Undefined               = 0x0300,
    OutputTermainalType_Speaker                 = 0x0301,
    OutputTermainalType_Headphones              = 0x0302,
    OutputTermainalType_HeadMountedDisplay      = 0x0303,
    OutputTermainalType_DesktopSpeaker          = 0x0304,
    OutputTermainalType_RoomSpeaker             = 0x0305,
    OutputTermainalType_Communication           = 0x0306,
    OutputTermainalType_LowFrequecyEffects      = 0x0307,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.4
 */
enum BidirectionalTerminalType
{
    BidirectionalTerminalType_Undefined         = 0x0400,
    BidirectionalTerminalType_Handset           = 0x0401,
    BidirectionalTerminalType_Headset           = 0x0402,
    BidirectionalTerminalType_Speakerphone      = 0x0403,
    BidirectionalTerminalType_EchoSuppressing   = 0x0404,
    BidirectionalTerminalType_EchoCancelling    = 0x0405,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.5
 */
enum TelephonyTerminalType
{
    TelephonyTerminalType_Undefined             = 0x0500,
    TelephonyTerminalType_PhoneLine             = 0x0501,
    TelephonyTerminalType_Telephone             = 0x0502,
    TelephonyTerminalType_DownLinePhone         = 0x0503,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.6
 */
enum ExternalTerminalType
{
    ExternalTerminalType_Undefined              = 0x0600,
    ExternalTerminalType_AnalogConnector        = 0x0601,
    ExternalTerminalType_DigitalAudioInterface  = 0x0602,
    ExternalTerminalType_LineConnector          = 0x0603,
    ExternalTerminalType_LegacyAudioConnector   = 0x0604,
    ExternalTerminalType_SPDIFInterface         = 0x0605,
    ExternalTerminalType_1395DAStream           = 0x0606,
    ExternalTerminalType_1394StreamSoundtrack   = 0x0607,
};


/**
 * @brief USB Device Class Definition for Terminal Types, Section 2.7
 */
enum EmbeddedTerminalType
{
    EmbeddedTerminalType_Undefined              = 0x0700,
    EmbeddedTerminalType_LevelCalibrationSource = 0x0701,
    EmbeddedTerminalType_EqualizationNoise      = 0x0702,
    EmbeddedTerminalType_CDPlayer               = 0x0703,
    EmbeddedTerminalType_DAT                    = 0x0704,
    EmbeddedTerminalType_DCC                    = 0x0705,
    EmbeddedTerminalType_MiniDisk               = 0x0706,
    EmbeddedTerminalType_AnalogTape             = 0x0707,
    EmbeddedTerminalType_Phonograph             = 0x0708,
    EmbeddedTerminalType_VCRAudio               = 0x0709,
    EmbeddedTerminalType_VideoDiscAudio         = 0x070a,
    EmbeddedTerminalType_DVDAudio               = 0x070b,
    EmbeddedTerminalType_TVTunerAudio           = 0x070c,
    EmbeddedTerminalType_SatelliteReceiverAudio = 0x070d,
    EmbeddedTerminalType_CableTunerAudio        = 0x070e,
    EmbeddedTerminalType_DSSAudio               = 0x070f,
    EmbeddedTerminalType_RadioReceiver          = 0x0710,
    EmbeddedTerminalType_RadioTransmitter       = 0x0711,
    EmbeddedTerminalType_MultitrackRecorder     = 0x0712,
    EmbeddedTerminalType_Sunthesizer            = 0x0713,
};


/**
 * @brief Universal Serial Bus Device Class Definition for Audio Devices, Section 3.7.2.3
 */
enum ChannelCluster
{
    ChannelCluster_LeftFront                    = 0x0001,
    ChannelCluster_RightFront                   = 0x0002,
    ChannelCluster_CenterFront                  = 0x0004,
    ChannelCluster_LowFrequencyEnhancement      = 0x0008,
    ChannelCluster_LeftSurround                 = 0x0010,
    ChannelCluster_RightSurround                = 0x0020,
    ChannelCluster_LeftCenter                   = 0x0040,
    ChannelCluster_RightCenter                  = 0x0080,
    ChannelCluster_Surround                     = 0x0100,
    ChannelCluster_SideLeft                     = 0x0200,
    ChannelCluster_SideRight                    = 0x0400,
    ChannelCluster_Top                          = 0x0800,
};


/**
 * @brief USB Device Class Definition for Audio Data Formats, Appendix A.1
 */
enum AudioDataFormat
{
    AudioDataFormat_TypeIUndefined              = 0x0000,
    AudioDataFormat_Pcm                         = 0x0001,
    AudioDataFormat_Pcm8                        = 0x0002,
    AudioDataFormat_IeeeFloat                   = 0x0003,
    AudioDataFormat_Alaw                        = 0x0004,
    AudioDataFormat_Mulaw                       = 0x0005,

    AudioDataFormat_TypeIIUndefined             = 0x1000,
    AudioDataFormat_TypeIIMpeg                  = 0x1001,
    AudioDataFormat_TypeIIAc3                   = 0x1002,

    AudioDataFormat_TypeIIIUndefined            = 0x2000,
    AudioDataFormat_Iec1937Ac3                  = 0x2001,
    AudioDataFormat_Iec1937Mpeg1Layer1          = 0x2002,
    AudioDataFormat_Iec1937Mpeg1Layer23         = 0x2003,
    AudioDataFormat_Iec1937Mpeg2Ext             = 0x2004,
    AudioDataFormat_Iec1937Mpeg2Layer1Ls        = 0x2005,
    AudioDataFormat_Iec1937Mpeg2Layer23Ls       = 0x2006,
};


/**
 * @brief Referenced by struct below, multi byte encoding for sampling frequency
 */
struct AudioSamplingFrequency
{

    uint8_t         n1;
    uint8_t         n256;
    uint8_t         n65536;

} __attribute__((packed));


/**
 * @brief USB Device Class Definition for Audio Data Formats, Section 2.2.5
 */
struct AudioStreamingFormatType
{

    AudioDescriptor         audioDescriptor;
    uint8_t                 bFormatType;
    uint8_t                 bNrChannles;
    uint8_t                 bSubFrameSize;
    uint8_t                 bBitResolution;
    uint8_t                 bSamFreqType;   // number of entries in following sampling frequency table
    AudioSamplingFrequency  tSamFreq[1];

} __attribute__((packed));




} // end of namespace cduac
} // end of namespace nn
