﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      cdacm Types declaration
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/cdacm/cdacm_Result.h>

namespace nn {
namespace cdacm {

/**
 * @brief Handle representing an enumerated USB-to-serial device.
 *        The lifecycle of this handle begins when a mass storage device is enumerated,
 *        and ends when it is logically disconnected. Value of zero is invalid.
 */
typedef uintptr_t UnitHandle;

/**
 * @brief Profile of a connected device.
 */
struct UnitProfile
{
    UnitHandle   handle;              // Handle of this device
    uint16_t     vid;                 // VID of the device
    uint16_t     pid;                 // PID of the device
};

/*
 * MODEM CONTROL:
 * B0    DTR state
 *          0 = reset
 *          1 = set
 * B1    RTS state
 *          0 = reset
 *          1 = set
 * B2..7 Reserved
 * B8    DTR state enable
 *          0 = ignore
 *          1 = use DTR state
 * B9    RTS state enable
 *          0 = ignore
 *          1 = use RTS state
 * B10..15 Reserved
 *
 * FLOW CONTROL:
 * hIndex protocol is:
 *   B0 Output handshaking using RTS/CTS
 *       0 = disabled
 *       1 = enabled
 *   B1 Output handshaking using DTR/DSR
 *       0 = disabled
 *       1 = enabled
 *   B2 Xon/Xoff handshaking
 *       0 = disabled
 *       1 = enabled
 *
 * A value of zero in the hIndex field disables handshaking
 *
 * If Xon/Xoff handshaking is specified, the hValue field should contain the
 * XOFF character and the lValue field contains the XON character.
 *
 * DATA CHARACTERISTICS:
 *   B0..7   Number of data bits
 *   B8..10  Parity
 *           0 = None
 *           1 = Odd
 *           2 = Even
 *           3 = Mark
 *           4 = Space
 *   B11..13 Stop Bits
 *           0 = 1
 *           1 = 1.5
 *           2 = 2
 *   B14
 *           1 = TX ON (break)
 *           0 = TX OFF (normal state)
 *   B15 Reserved
 *
 * BAUDRATE:
 * The following are valid: 1200, 2400, 4800, 9600, 19200, 38400, 57600, 1152000
 *
 * EVENT CHARACTER:
 *   B0..7   Event Character
 *   B8      Event Character Processing
 *             0 = disabled
 *             1 = enabled
 *   B9..15  Reserved
 *
 * ERROR CHARACTER:
 *  B0..7  Error Character
 *  B8     Error Character Processing
 *           0 = disabled
 *           1 = enabled
 *  B9..15 Reserved
 *
 * LATENCY:
 *   B0..7   Latency timer
 *   B8..15  0
 */

struct CdAcmParameters
{
    uint16_t modemControl;
    uint16_t flowCtrl;
    uint16_t dataCharacteristics;
    uint32_t baudRate;
    uint16_t eventCharacter;
    uint16_t errorCharacter;
    uint16_t latency;
};

}  // cdacm
}  // nn
