﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ユーザアプリケーション向け API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>

#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/user/btm_UserResult.h>

/**
 * @namespace   nn::btm::usr
 * @brief       ユーザアプリケーション向け btm API の名前空間です。
 */

namespace nn { namespace btm { namespace user {
//! @name BLE Scan 機能
//! @{

/**
 * @brief       BTM のIPC サーバとのセッションを構築します
 * @details     同期API です。
 *              セッションはライブラリ内で保持されます。他のすべてのAPI よりも先に呼び出しておく必要があります。
 *              FinalizeBtmUserInterface() の呼び出しを挟まない、2回目以降の呼び出しはできません。
 *
 * @pre
 *              - セッションが構築されていない
 */
void InitializeBtmUserInterface() NN_NOEXCEPT;

/**
 * @brief       BTM のIPC サーバとのセッションを破棄します
 * @details     同期API です。
 *              セッション破棄後は、再度セッションを取得するまでSF 機能以外のAPI は使用できません。
 *              InitializeBtmUserInterface() の呼び出しを挟まない、2回目以降の呼び出しはできません。
 *
 * @pre
 *              - セッションが構築されている
 */
void FinalizeBtmUserInterface() NN_NOEXCEPT;

/**
 * @brief       BLE Scan の状態が変わった際にシグナルするシステムイベントを取得します。
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleScanEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE スキャンのフィルタパラメータを取得します。
 *
 * @details     StartBleScanForGeneral() を使用してBLE デバイスをスキャンするために必要なパラメータを取得します。
 *              引数parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pFilter                         スキャンフィルタオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - 初期化済み
 *              - pFilter != nullptr
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::user::ResultInvalidArgument,         引数の値が無効です。}
 * @endretresult
 */
nn::Result GetBleScanFilterParameter(BleAdvFilterForGeneral* pFilter, uint16_t parameterId) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスアプリを対象としたBLE スキャンのフィルタパラメータを取得します。
 *
 * @details     StartBleScanForSmartDevice() を使用してスマートデバイスアプリをスキャンするために必要なパラメータをシステムから取得します。
 *              引数parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pFilter                         スキャンフィルタオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - 初期化済み
 *              - pFilter != nullptr
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::user::ResultInvalidArgument,         引数の値が無効です。}
 * @endretresult
 */
nn::Result GetBleScanFilterParameter(BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイス以外のBLE デバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              設定されたフィルタはStopBleScanGeneral() が呼ばれるまで有効であり、有効である間スキャンを継続します。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanGeneral() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、AcquireBleScanEvent() を使って取得したシステムイベントがシグナルされます。
 *              スキャン結果は、GetBleScanResultsForGeneral() を使って取得してください。
 *
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,            ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral, 下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::user::ResultScanStartFailedFull,    スキャン対象が最大に達しています。}
 * @endretresult
 */
nn::Result StartBleScanForGeneral(const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForGeneral() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForGeneral() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,            ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral, 下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForGeneral() NN_NOEXCEPT;

/**
 * @brief       BLE Scan で発見されたデバイスを取得します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForGeneral() を使用して開始されたスキャンで発見されたデバイスを取得します。
 *
 *
 * @param[out]  pResults                        スキャン結果を格納するための配列へのポインタ
 * @param[in]   inNum                           pResults の配列サイズ
 *
 * @pre
 *              - 初期化済み
 *              - pResults != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pResults にスキャン結果が格納されている。
 *
 * @return      取得したスキャン結果の数
 */
uint8_t GetBleScanResultsForGeneral(ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT;

/**
 * @brief       ペアリングされたBLE デバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイス以外の、ペアリングされたBLE デバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              設定されたフィルタはStopBleScanGeneral() が呼ばれるまで有効であり、有効である間スキャンを継続します。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanForPaired() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、接続を試行します。
 *              接続操作が完了すると、AcquireBleConnectionEvent() で取得したイベントがシグナルされます。
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                   処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,                    ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral,         下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::user::ResultScanStartFailedFull,            スキャン対象が最大に達しています。}
 *   @handleresult{nn::btm::user::ResultScanStartFailedPairingNotFound, ペアリング済みBLE デバイスのスキャンを開始しようとしましたが、ペアリング情報が存在しません。}
 * @endretresult
 */
nn::Result StartBleScanForPaired(const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされたBLE デバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForPaired() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForPaired() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,            ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral, 下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForPaired() NN_NOEXCEPT;

/**
 * @brief       スマートデバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanForSmartDevice() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、AcquireBleScanEvent() を使って取得したシステムイベントがシグナルされます。
 *              スキャン結果は、GetBleScanResultsForSmartDevice() を使って取得してください。
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,            ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral, 下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::user::ResultScanStartFailedFull,    スキャン対象が最大に達しています。}
 * @endretresult
 */
nn::Result StartBleScanForSmartDevice(const BleAdvFilterForSmartDevice &filter) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForSmartDevice() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForSmartDevice() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,            ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultFailureLowLayerGeneral, 下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForSmartDevice() NN_NOEXCEPT;

/**
 * @brief       BLE Scan で発見されたデバイス一覧を取得します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForSmartDevice() を使用して開始されたスキャンで発見されたデバイスを取得します。
 *              受信電波強度が強い順に最大でnn::bluetooth::BLE_SCAN_RESULT_NUM_MAX 個取得することができます。
 *
 *
 * @param[out]  pResults                        スキャン結果を格納するための配列へのポインタ
 * @param[in]   inNum                           pResults の配列サイズ
 *
 * @pre
 *              - 初期化済み
 *              - pResults != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pResults にスキャン結果が格納されている。スキャン結果が存在しない場合はnullptr。
 *
 * @return      取得したスキャン結果の数
 */
uint8_t GetBleScanResultsForSmartDevice(ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT;

//! @}

//! @name BLE 接続機能
//! @{

/**
 * @brief       BLE の接続状態が変わった際にシグナルするシステムイベントを取得します。
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       BLE の接続を試行します。
 * @details     ノンブロッキングAPIです。
 *              指定されたアドレスを持つデバイスに対し接続を試行します。
 *              接続の結果は、AcquireBleConnectionEvent() を使用して取得したシステムイベントを通じて行われます。
 *
 * @param[in]   address                         接続を試行するBLE デバイス、スマートデバイスのBluetooth アドレス
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBleConnectionAlreadyExist,          すでに接続しています。}
 *   @handleresult{nn::btm::user::ResultBleConnectionFailedFull,            接続台数が最大に達しています。}
 *   @handleresult{nn::btm::user::ResultInvalidOperationForCurrentUsecase,  現在のユースケース上では許可されていない操作です。適切なユースケースに遷移してください。}
 * @endretresult
 */
nn::Result BleConnect(nn::bluetooth::Address address) NN_NOEXCEPT;

/**
 * @brief       BLE の接続を切断します。
 * @details     ノンブロッキングAPIです。
 *              指定された接続ハンドルを持つBLE 接続を切断します。
 *              切断の結果は、AcquireBleConnectionEvent() を使用して取得したシステムイベントを通じて行われます。
 *
 * @param[in]   connectionHandle                 切断するBLE 接続の接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBleConnectionNotFound,  接続が存在しません。}
 * @endretresult
 */
nn::Result BleDisconnect(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       BLE の接続状態を返します。
 * @details     ブロッキングAPIです。
 *              最大でnn::bluetooth::BLE_GATT_CLIENT_NUM_MAX 個の接続状態を取得することができます。
 *
 * @param[out]  pConnState                      接続状態を格納するための配列へのポインタ
 * @param[in]   inNum                           pConnState の配列のサイズ
 *
 * @pre
 *              - 初期化済み
 *              - pConnState != nullptr
 *              - inNum != 0
 *
 * @return      取得した接続状態
 */
uint8_t BleGetConnectionState(user::BleClientConnState *pConnState, uint8_t inNum) NN_NOEXCEPT;

//! @}

//! @name GATT Service Discovery に関する機能
//! @{

/**
 * @brief       GATT Service Disvery が完了した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 *              BLE 接続が確立した後、自動的にGATT Service Discovery が実行され、GATT Attribute が見つかる度にシグナルされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server のGatt Service の一覧を取得します
 * @details     ブロッキングAPI です。
 *              GATT Service Discovery の完了前に呼び出すと、不完全なGatt Service の一覧が返る可能性があります。
 *
 * @param[out]  pServices                       取得したGatt Service を格納するための配列へのポインタ
 * @param[in]   inNum                           pServices の配列サイズ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pServices != nullptr
 *
 * @post
 *              - pServices にはGatt Service 一覧が格納されている
 *
 * @return      取得したGATT Service の数
 */
uint8_t GetGattServices(GattService *pServices, uint8_t inNum, uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の、指定したGatt Service に属するGatt Included Service の一覧を取得します
 * @details     ブロッキングAPI です。
 *              GATT Service Discovery の完了前に呼び出すと、不完全なGatt Included Service の一覧が返る可能性があります。
 *
 * @param[out]  pIncludedServices               取得したGatt Included Service を格納するための配列へのポインタ
 * @param[in]   inNum                           pIncludedServices の配列サイズ
 * @param[in]   connectionHandle                Gatt Included Service を取得するGatt Server との接続ハンドル
 * @param[in]   serviceHandle                   Gatt Included Service を取得するGatt Service のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pIncludedServices != nullptr
 *
 * @post
 *              - pIncludedServices にはGatt Service 一覧が格納されている
 *
 * @return      取得したGATT Service の数
 */
uint8_t GetGattIncludedServices(GattService *pIncludedServices, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の、指定したUUID を持つGatt Service を取得します
 * @details     ブロッキングAPI です。
 *
 * @param[out]  pService                        Gatt Service オブジェクトへのポインタ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 * @param[in]   uuid                            取得するGatt Service のUUID
 *
 * @pre
 *              - 初期化済み
 *              - pService != nullptr
 *
 * @post
 *              - *pService にはGatt Service が格納されている
 *
 * @return      Gatt Service が取得できた場合はtrue。そうでない場合は、false。
 */
bool GetGattService(GattService *pService, uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の、指定したHandle を持つGatt Attribute が所属するGatt Service を取得します
 * @details     ブロッキングAPI です。
 *
 * @param[out]  pService                        Gatt Service オブジェクトへのポインタ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 * @param[in]   attributeHandle                 Gatt Service を取得するGatt Attribute のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pService != nullptr
 *
 * @post
 *              - *pService にはGatt Service が格納されている
 *
 * @return      Gatt Service が取得できた場合はtrue。そうでない場合は、false。
 */
bool GetBelongingGattService(GattService *pService, uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したGatt Service に属するGatt Characteristic の一覧を取得します
 * @details     ブロッキングAPI です。
 *               GATT Service Discovery の完了前に呼び出すと、不完全なGatt Characteristic の一覧が返る可能性があります。
 *
 * @param[out]  pCharacteristics                取得したGatt Characteristic を格納するための配列へのポインタ
 * @param[in]   inNum                           pCharacteristics の配列サイズ
 * @param[in]   connectionHandle                Gatt Characteristic を取得するGatt Server との接続ハンドル
 * @param[in]   serviceHandle                   Gatt Characteristic を取得するGatt Service のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pCharacteristics != nullptr
 *
 * @post
 *              - pCharacteristics にはGatt Characteristic 一覧が格納されている
 *
 * @ret         取得したGATT Characteristic の数
 */
uint8_t GetGattCharacteristics(GattCharacteristic *pCharacteristics, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したGatt Characteristic に属するGat Descriptor の一覧を取得します
 * @details     ブロッキングAPI です。
 *              GATT Service Discovery の完了前に呼び出すと、不完全なGatt Descriptor の一覧が返る可能性があります。
 *
 * @param[out]  pDescriptors                    取得したGatt Descriptor を格納するための配列へのポインタ
 * @param[in]   inNum                           pDescriptors の配列サイズ
 * @param[in]   connectionHandle                Gatt Descriptor を取得するGatt Server との接続ハンドル
 * @param[in]   charHandle                      Gatt Descriptor を取得するGatt Characteristic のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pDescriptors != nullptr
 *
 * @post
 *              - pDescriptors にはGatt Descriptor 一覧が格納されている
 *
 * @ret         取得したGATT Descriptor の数
 */
uint8_t GetGattDescriptors(GattDescriptor *pDescriptors, uint8_t inNum, uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT;

//! @}

//! @name BLE ペアリング機能
//! @{

/**
 * @brief       BLE ペアリング処理が完了または処理中にエラーが発生した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBlePairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       接続されているGATT Server とペアリングします。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              最大nn::bluetooth::BLE_PAIRING_NUM_MAX 個のGATT Server とペアリングすることができます。
 *              ペアリング数が最大に達している場合、最も古いペアリング情報を削除します。
 *              すでにペアリング情報が存在する場合、本体内のペアリング情報を更新します。
 *              ペアリング処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   connectionHandle                ペアリングするGATT Server との接続ハンドル
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultBleConnectionNotFound,         接続が存在しません。}
 *  @handleresult{nn::btm::ResultBlePairingNotSupported,        ペアリング機能がサポートされていません。}
 * @endretresult
 */
nn::Result BlePairDevice(uint32_t connectionHandle, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       接続されているGATT Server とのペアリング情報を削除します。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              ペアリング情報が存在しないでも、GATT Server からペアリング情報を削除する処理を行います。
 *              ペアリング削除処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   connectionHandle                ペアリングを削除するGATT Server との接続ハンドル
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::user::ResultBleConnectionNotFound,   接続が存在しません。}
 *  @handleresult{nn::btm::user::ResultBlePairingNotSupported,  ペアリング機能がサポートされていません。}
 * @endretresult
 */
nn::Result BleUnPairDevice(uint32_t connectionHandle, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       GATT Server とのペアリングを削除します。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              指定したBluetooth アドレスを持つGATT Server と接続している場合、本体、及び、GATT Server からペアリング情報を削除する処理を行います。
 *              指定したBluetooth アドレスを持つGATT Server と接続していない場合は、本体からのみペアリング情報を削除します。
 *              ペアリング情報が存在しない場合でも、指定したBluetooth アドレスを持つGATT Server と接続されいる場合、そのGATT Server からペアリング情報を削除する処理を行います。
 *              ペアリング削除処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   bluetoothAddress                ペアリングを削除するGATT Server のBluetooth アドレス
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::user::ResultPairingNotSupported,     ペアリング機能がサポートされていません。}
 * @endretresult
 */
nn::Result BleUnPairDevice(const nn::bluetooth::Address& address, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされているGATT Server のBluetooth アドレス一覧を取得します。
 *
 * @details     最大でBLE_PAIRING_NUM_MAX 個のBluetooth アドレスを取得します。
 *
 * @param[out]  pOutAddresses                   ペアリングされたGATT Server のBluetooth アドレスを格納するための配列へのポインタ
 * @param[in]   count                           pOutAddresses の数
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @pre
 *              - pOutAddresses != nullptr
 *
 * @return      取得したBluetooth アドレスの数
 *
 */
uint8_t BleGetPairedDevices(nn::bluetooth::Address* pOutAddresses, uint8_t count, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

//! @}


//! @name その他のBLE 関連機能
//! @{

/**
 * @brief       GATT Server との間でMTU 設定が完了した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Server との間で、MTU 設定を行います。
 * @details     ノンブロッキングAPI です。
 *              MTU 設定が完了すると、AcquireBleMtuConfigEvent() で取得したイベントがシグナルされます。
 *              設定可能なMTU は最小で23 Byte, 最大で512 Byte です。
 *              同時に MTU を設定可能な接続は 1 つまでです。
 *
 * @param[in]   connectionHandle                MTU を設定するGATT Server との接続ハンドル
 * @param[in]   mtu                             設定するMTU
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBleConnectionNotFound,      接続が存在しません。}
 *   @handleresult{nn::btm::user::ResultBusyProcessingExlusiveApi,  排他的な API の処理中です。処理を待ってから再試行してください。}
 * @endretresult
 */
nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT;

/**
 * @brief       GATT Server との間のMTU を取得します。
 * @details     ブロッキングAPI です。
 *              AcquireBleMtuConfigEvent() で取得したシステムイベントがシグナルされる前に呼び出すと、設定変更前のMTU が返る可能性があります。
 *              接続が見つからない場合、0 を返します。
 *
 * @param[in]   connectionHandle                MTU を取得するGATT Server との接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *
 * @return      取得したMTU
 */
uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知先を登録します
 *
 * @details     指定したUUID を持つGATT Service に属するGATT Attribute を操作した際の結果に関するシステムイベントの通知先を登録します。
 *
 * @param[in]   uuid                            通知先を登録するGATT Service のUUID
 * @param[in]   path                            イベントの通知先識別子
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                   処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,                    ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::user::ResultBleGattDataPathRegisterFailed,  GATT 操作の通知先の登録が最大数に達しているか、すでに登録されています。}
 * @endretresult
 * @
 */
nn::Result RegisterBleGattDataPath(const BleDataPath& path) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知先の登録を削除します
 *
 * @details     RegisterBleGattDataPath() を使用して登録された通知を削除します。
 *
 * @param[in]   uuid                            通知先登録を削除するGATT Service のUUID
 * @param[in]   path                            イベントの通知先識別子
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::user::ResultBusyWorking,                ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 * @endretresult
 * @
 */
nn::Result UnregisterBleGattDataPath(const BleDataPath& path) NN_NOEXCEPT;

//! @}

}}} // namespace nn::btm::user

