﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file   bluetooth_Api.h
 * @brief  Bluetooth API
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

namespace nn { namespace bluetooth { namespace user {
/**
 * @brief  Initialize the service of bluetooth low energy.
 *
 * @details
 * NON_BLOCKING.
 * This API initialize functions for the SF.
 * This API should be called before any other bluetooth low engery APIs.
 *
 * @callback None
 *
 * @retval void.
 *
 */
void InitializeBluetoothUserInterface() NN_NOEXCEPT;

/**
 * @brief  Finalize the service of bluetooth low energy.
 *
 * @details
 * NON-BLOCKING.
 * This API finalize functions for the SF.
 * This API is pairs with "InitializeBleInterface".
 * This API should be called last when shutting down.
 *
 * @retval void.
 *
 */
void FinalizeBluetoothUserInterface() NN_NOEXCEPT;

/**
* @brief  Register BLE events.
*
* @details
* BLOCKING.
* Attach a system event for notification of BLE general GATT operation.
*
* @param[in] pSystemEvent Hid event to be registered.
*
* @retval ResultSuccess if okay, otherwise ResultFail.
*
*/
nn::Result RegisterBleEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief  Read a characteristic
 *
 * @details
 * BLOCKING
 * Read a characteristic on a remote device
 *
 * @callback EventFromLeReadCharacteristicCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_characteristic)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, int auth_req ); Android API example

/**
 * @brief  Read a descriptor
 *
 * @details
 * BLOCKING
 * Read the descriptor for a given characteristic
 *
 * @callback EventFromLeReadDescriptorCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_descriptor)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, btgatt_gatt_id_t *descr_id, int auth_req); Android API example

/**
 * @brief  Write a characteristic
 *
 * @details
 * BLOCKING
 * Write a characteristic on a remote device
 *
 * @callback EventFromLeWriteCharacteristicCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse) NN_NOEXCEPT;
//bt_status_t (*write_characteristic)(int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, int write_type, int len, int auth_req, char* p_value); Android API example

/**
 * @brief  Write a descriptor
 *
 * @details
 * BLOCKING
 * Write the descriptor for a given characteristic
 *
 * @callback EventFromLeWriteDescriptorCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType) NN_NOEXCEPT;
//bt_status_t (*read_descriptor)( int conn_id, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id, btgatt_gatt_id_t *descr_id, int auth_req); Android API example

/**
 * @brief  Register a notification
 *
 * @details
 * BLOCKING
 * Register to receive notifications or indications for a given
 * characteristic from a service
 *
 * @callback EventFromLeRegisterNotificationCallback
 * @callback EventFromLeNotificationCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT;
//bt_status_t (*register_for_notification)( int client_if, const bt_bdaddr_t *bd_addr, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id); Android API example

/**
 * @brief  Deregister a notification
 *
 * @details
 * BLOCKING
 * Deregister a previous request for notifications/indications
 *
 * @callback EventFromLeDeregisterNotificationCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT;
//bt_status_t (*deregister_for_notification)( int client_if, const bt_bdaddr_t *bd_addr, btgatt_srvc_id_t *srvc_id, btgatt_gatt_id_t *char_id); Android API example

/**
 * @brief  Add a BLE GATT descriptor to a service.
 *
 * @details
 * BLOCKING
 *
 * @callback EventFromSendLeIndicationCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result LeSendIndication(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, const uint8_t* pData, uint8_t len, bool confirm) NN_NOEXCEPT;
//bt_status_t (*send_indication)(int server_if, int attribute_handle, int conn_id, int len, int confirm, char* p_value); Android API example
/**
 * @brief  Set response to a BLE GATT characteristic or descriptor
 *
 * @details
 * BLOCKING
 * This function sends the server response to client.
 *
 * @callback EventFromSendLeResponseCallback
 *
 * @retval ResultSuccess.
 *
 */
nn::Result SetLeResponse(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const uint8_t *pData, uint16_t len) NN_NOEXCEPT;
//bt_status_t (*send_response)(int conn_id, int trans_id, int status, btgatt_response_t *response); Android API example

/**
 * @brief  Get the information BLE general event info from the Server.
 *
 * @details
 * BLOCKING
 *
 * @param[out] pOutEventType Event type.
 * @param[out] outBuffer     Event Data.
 * @param[in]  sizeOfBuffer  Size of event data.
 *
 * @retval ResultSuccess.
 *
 */
nn::Result GetLeEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT;

} // namespace user
} // namespace bluetooth
} // namespace nn
