﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       BLE に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/bluetooth/bluetooth_LeTypes.h>
#include <nn/bluetooth/bluetooth_Result.h>

#include <nn/bluetooth/bluetooth_GattService.h>
#include <nn/bluetooth/bluetooth_GattCharacteristic.h>
#include <nn/bluetooth/bluetooth_GattDescriptor.h>

namespace nn { namespace bluetooth {

//! @name 一般的な BLE 機能に関する API
//! @{

/**
 * @brief       BLE ライブラリを初期化して使用可能にします。
 *
 * @details     他の BLE に関する API よりも先に呼び出しておく必要があります。
 */
void InitializeBle() NN_NOEXCEPT;

/**
 * @brief       BLE スキャン中にアドバタイズパケットを発行した際にシグナルするシステムイベントを取得します。
 *
 * @details     本 API で取得されるシステムイベントは、BLE スキャンで新規にデバイスを発見した場合だけでなく、
 *              過去に発見したアドバタイズパケットが（RSSI やアドバタイズパケット内のデータなど）何らか変化した場合にもシグナルされます。
 *              対象となる BLE スキャンは、以下 API を使用して開始されたものです。
 *              - StartBleScanGeneral()
 *              - StartBleScanSmartDevice()
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBleScanEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象とした BLE スキャンを開始するためのパラメータを取得します。
 *
 * @details     StartBleScanGeneral() を使用して BLE デバイスをスキャンするために必要なパラメータをシステムから取得します。
 *              parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pOutParameter                   取得するパラメータを格納するためのオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - pOutManufacturerId != nullptr
 */
void GetBleScanParameter(BleAdvertisePacketParameter* pOutParameter, int parameterId) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスアプリを対象とした BLE スキャンを開始するためのパラメータを取得します。
 *
 * @details     StartBleScanSmartDevice() を使用してスマートデバイスアプリをスキャンするために必要なパラメータをシステムから取得します。
 *              parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pOutUuid                        指定のアドバタイズパケットフォーマットのうちの 128 bit Service UUID を表すオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - pOutUuid != nullptr
 */
void GetBleScanParameter(GattAttributeUuid* pOutUuid, int parameterId) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象とした BLE スキャンを開始します。
 *
 * @details     開始された BLE スキャンは StopBleScanGeneral() を呼ぶまで継続します。
 *              アドバタイズパケットの発見時に AcquireBleScanEvent() で取得したイベントがシグナルされます。
 *              複数種類の BLE デバイスをスキャンしたい場合、スキャンしたい BLE デバイスに合わせたパラメータを使用して、本 API を複数回呼び出してください。
 *              他の BLE スキャンと同時に使用できます。
 *              異なるスキャンを並行して行っているかに関わらず、本 API によって同時にスキャン可能な BLE デバイスは 5 種類までです。
 *
 * @param[in]   scanParameter                   発見したい BLE デバイスの BLE アドバタイズパケットのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleScanFull,             スキャン対象数の上限に達しています。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result StartBleScanGeneral(const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象とした BLE スキャンを停止します。
 *
 * @details     StartBleScanGeneral() を複数回呼び出して、複数種類の BLE デバイスをスキャンしている場合、それらのスキャンが全て停止します。
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result StopBleScanGeneral();

/**
 * @brief       スマートデバイスアプリケーションを対象とした BLE スキャンを開始します。
 *
 * @details     開始された BLE スキャンは StopBleScanSmartDevice() を呼ぶまで継続します。
 *              アドバタイズパケットの発見時に AcquireBleScanEvent() で取得したイベントがシグナルされます。
 *              複数種類のスマートデバイスアプリケーションをスキャンしたい場合、
 *              スキャンしたいスマートデバイスアプリケーションに合わせたパラメータを使用して、本 API を複数回呼び出してください。
 *              他の BLE スキャンと同時に使用できます。
 *              異なるスキャンを並行して行っているかに関わらず、本 API によって同時にスキャン可能な BLE デバイスは 5 種類までです。
 *
 * @param[in]   uuid                             指定のアドバタイズパケットフォーマットのうちの 128 bit Service UUID
 *
 * @pre
 *              - uuid.length == @ref GattAttributeUuidLength_128
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleScanFull,             スキャン対象数の上限に達しています。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 *
 */
nn::Result StartBleScanSmartDevice(const GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスアプリケーションを対象とした BLE スキャンを停止します。
 *
 * @details     StartBleScanSmartDevice() を複数回呼び出して、複数種類のスマートデバイスアプリケーションをスキャンしている場合、それらのスキャンが全て停止します。
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result StopBleScanSmartDevice() NN_NOEXCEPT;

/**
* @brief       StartBleScanGeneral() または StartBleScanSmartDevice() を使用して開始された BLE スキャンの結果一覧を取得します。
*
* @details     最大で @ref BleScanResultCountMax 個のスキャン結果を得られます。
*              スキャン結果は RSSI の降順にソートされています。
*              BLE スキャンが行われていない場合、スキャン結果は空になります。
*              BLE スキャンを開始する度に、それぞれに対応したスキャン結果のみがクリアされます。
*
* @param[out]  pOutScanResult                  スキャン結果を格納するための配列へのポインタ
* @param[in]   count                           pOutScanResult の数
*
* @pre
*              - pOutScanResult != nullptr
*
* @post
*              - pOutScanResult にはスキャン結果が格納されている
*/
int GetBleScanResult(BleScanResult* pOutScanResult, int count) NN_NOEXCEPT;

/**
 * @brief       GATT Server との接続状態が変化した場合にシグナルするシステムイベントを取得します。
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBleConnectionStateChangedEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Server に接続します。
 *
 * @details     接続完了後に AcquireBleConnectionStateChangedEvent() で取得したイベントがシグナルされます。
 *              GetBleConnectionState() を使用して、接続されている GATT Server の一覧を取得することができます。
 *
 * @param[in]  bdAddress                       接続する GATT Server のBluetooth アドレス
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBleAlreadyConnected,     すでに接続が存在します。}
 *  @handleresult{nn::bluetooth::ResultBleConnectionFull,       接続数が最大に達しています。}
 *  @handleresult{nn::bluetooth::ResultInvalidWirelessState,    他の無線機能の使用状況が原因で、操作を実行できません。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result ConnectToGattServer(const Address& bdAddress) NN_NOEXCEPT;

/**
 * @brief       GATT Server から切断します。
 *
 * @details     切断完了後に AcquireBleConnectionStateChangedEvent() で取得したイベントがシグナルされます。
 *              GetBleConnectionState() を使用して、接続されている GATT Server の一覧を取得することができます。
 *
 * @param[in]   connectionHandle                切断するGATT Server との接続のハンドル
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 * @endretresult
 */
nn::Result DisconnectFromGattServer(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Server との接続情報の一覧を取得します。
 *
 * @details     最大で @ref BleConnectionCountMaxClient 個の接続状態を取得します。
 *
 * @param[out]  pOutConnectionInfo              GATT Server との接続情報を格納する配列へのポインタ
 * @param[in]   count                           pOutConnectionInfo の数
 *
 * @pre
 *              - pOutConnectionInfo != nullptr
 *
 * @post
 *              - pOutConnectionInfo には現在の GATT Server との接続情報の一覧が格納されている
 *
 * @return      取得した接続情報のうち、GATT Server と接続されている接続の数
 */
int GetBleConnectionInfoList(BleConnectionInfo* pOutConnectionInfo, int count) NN_NOEXCEPT;

/**
 * @brief       GATT Service Discovery が行われた際にシグナルされるイベントを取得します。
 *
 * @details     GATT Server との接続後、自動的に GATT Service Discovery が実行され、GATT Server が提供する GATT Attribute をすべて取得したタイミングで、
 *              本API で取得したイベントがシグナルされます。
 *              取得した GATT Attribute は以下の API を使用して取得することができます。
                - GetGattServices()
                - GattService::GetIncludedServices()
                - GattService::GetCharacterisitcs()
                - GattService::GetCharacteristic()
                - GattCharacteristic::GetService()
                - GattCharacteristic::GetDescriptors()
                - GattCharacteristic::GetDescriptor()
                - GattDescriptor::GetService()
                - GattDescriptor::GetCharacteristic()
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Server が提供する GATT Service の一覧を取得します。
 *
 * @details     最大で @ref GattAttributeCountMaxClient 個の GATT Service を取得します。
 *
 * @param[out]  pOutGatttServices               GATT Service を格納するための配列へのポインタ
 * @param[in]   count                           pOutGatttServices の数
 * @param[in]   connectionHandle                GATT Server との接続ハンドル
 *
 * @pre
 *              - pOutGatttServices != nullptr
 *
 * @post
 *              - pOutGatttServices に取得した GATT Service の一覧が格納されている
 *
 * @return      取得した GATT Service の数
 */
int GetGattServices(GattService* pOutGatttServices, int count, uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Server が提供する GATT Service のうち、指定の UUID を持つものを取得します。
 *
 * @param[out]  pOutGatttService                GATT Service オブジェクトへのポインタ
 * @param[in]   uuid                            取得したい GATT Service の UUID
 * @param[in]   connectionHandle                GATT Server との接続ハンドル
 *
 * @pre
 *              - pOutGatttService != nullptr
 *
 * @return      GATT Service が見つかれば true、見つからなければ false
 */
bool GetGattService(GattService* pOutGatttService, const GattAttributeUuid& uuid, uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Server との間で MTU 設定が行われた際にシグナルされるイベントを取得します。
 *
 * @details     ConfigureBleMtu() を使用して GATT Server との間の MTU 設定変更を要求したのち、実際に MTU が更新された際にイベントがシグナルされます。
 *              変更された MTU は GetBleMtu() を使用して取得してください。
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Server との MTU を設定します。
 *
 * @details     設定可能な MTU の最小値は @ref BleMtuDefault Byte、最大値は @ref BleMtuMax Byte ですが、実際に設定される MTU は GATT Server の仕様にも依存します。
 *              MTU の設定が変更されると、AcquireBleMtuConfigEvent() でバインドされたイベントがシグナルされます。
 *              同時に MTU を設定可能な接続は 1 つまでです。
 *
 * @param[in]   connectionHandle                MTU を設定する接続のハンドル
 * @param[in]   mtu                             設定する MTU
 *
 * @pre
 *              - @ref BleMtuDefault <= mtu <= @ref BleMtuMax
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultProcessingExclusiveApi,  排他的な API の処理中です。処理を待ってから再試行してください。}
 * @endretresult
 */
nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT;

/**
 * @brief       GATT Server との MTU を取得します。
 *
 * @details     接続が存在しない場合、0 が返ります。
 *
 * @param[in]   connectionHandle                MTU を設定する接続のハンドル
 *
 * @return      取得したMTU
 */
uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute の操作が完了した際にシグナルするシステムイベントを取得します。
 *
 * @details     本 API で取得されるシステムイベントは、以下の場合にシグナルされます。
 *              - GATT Attribute の Read、Write の操作を行った
 *              - GATT Attribute から Notification、Indication によってデータを受信した
 *
 *              イベントがシグナルされた場合、必ず BleGattOperationResult() を使用して、操作結果を取得してください。
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBleGattOperationEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知を受ける GATT Service を設定します
 *
 * @details     GATT Attribute に対する操作を行うには、対象の GATT Attribute が所属する GATT Service の UUID を本 API で登録する必要があります。
 *              本API を使用して登録していない GATT Service に所属する GATT Attribute に対して操作してはいけません。
 *              最大 @ref GattAttributeCountMaxClient 個の GATT Service を登録できます。
 *
 * @param[in]   uuid                            登録する GATT Service の UUID
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultDatabaseFull,            データベースの最大登録数に達しています。}
 * @endretresult
 */
nn::Result RegisterGattOperationNotification(const GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知を受ける GATT Service の設定を解除します
 *
 * @details     RegisterGattOperationNotification() による登録を解除します。
 *
 * @param[in]   uuid                            登録を削除する GATT Service の UUID
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 * @endretresult
 */
nn::Result UnregisterGattOperationNotification(const GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute の操作結果を取得します。
 *
 * @details     AcquireBleGattOperationEvent() を使用して取得したイベントがシグナルされた場合、本 API を使用して操作結果を必ず取得してください。
 *              AcquireBleGattOperationEvent() を使用して取得したイベントがシグナルされていない状態で本 API を使用すると、
 *              BleClientGattOperationInfo.status = @ref BleGattOperationStatus_OperationResultNotFound ,
 *              BleClientGattOperationInfo.operation = @ref GattOperationType_Unknown ,
 *              という操作結果が返ります。
 *
 *
 * @param[out]  pOutResult                      操作結果を格納するための構造体オブジェクトへのポインタ
 *
 * @pre
 *              - pOutResult != nullptr
 *
 * @post
 *              - pOutResult はGATT Attribute の操作結果
 */
void GetGattOperationResult(BleClientGattOperationInfo* pOutResult) NN_NOEXCEPT;

/**
 * @brief       GATT Characteristic を Read します
 *
 * @details     RegisterGattOperationNotification()  を使用して登録していない GATT Service に含まれる GATT Characteristic に対して操作してはいけません。
 *              API の呼び出しが成功すると、実際に通信が行われます。
 *              通信の完了後、AcquireBleGattOperationEvent() で取得したイベントがシグナルされます。
 *              操作結果の operation には、@ref GattOperationType_ReadCharacteristic が設定されています。
 *              受信したデータは、操作結果に含まれています。GattCharacteristic::GetValue() を使用して確認することはできません。
 *              イベントがシグナルされるまで、同一の接続に対して GATT Attribute を操作することはできません。
 *
 * @param[in]   characteristic                  Read する Gatt Characteristic のオブジェクト
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            操作に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultUnsupportedGattProperty, サポートされていないGATT Attribute Property です。}
 *  @handleresult{nn::bluetooth::ResultGattBusy,                ビジー状態です。直前の操作の完了を待って再試行してください。}
 * @endretresult
 */
nn::Result ReadGattCharacteristic(const GattCharacteristic& characteristic) NN_NOEXCEPT;

/**
 * @brief       GATT Characteristic に Write します
 *
 * @details     RegisterGattOperationNotification()  を使用して登録していない GATT Service に含まれる GATT Characteristic に対して操作してはいけません。
 *              GattCharacteristic::SetValue() を使用して設定した値を Write します。
 *              API の呼び出しが成功すると、実際に通信が行われます。
 *              通信の完了後、AcquireBleGattOperationEvent() で取得したイベントがシグナルされます。
 *              操作結果の operation には、@ref GattOperationType_WriteCharacteristic が設定されています。
 *              送信したデータは、操作結果に含まれていません。
 *              イベントがシグナルされるまで、同一の接続に対して GATT Attribute を操作することはできません。
 *
 * @param[in]   characteristic                  Write する Gatt Characteristic のオブジェクト
 *
 * @pre
 *              - GattCharacteristic::SetValue() を使用して、Write する値が設定されている
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            操作に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultUnsupportedGattProperty, サポートされていない GATT Attribute Property です。}
 *  @handleresult{nn::bluetooth::ResultGattBusy,                ビジー状態です。直前の操作の完了を待って再試行してください。}
 * @endretresult
*/
nn::Result WriteGattCharacteristic(const GattCharacteristic& characteristic) NN_NOEXCEPT;

/**
 * @brief       GATT Characteristic の Notification または Indication を有効化／無効化します
 *
 * @details     与えられた GATT Characteristic の Notification および Indication の受信を有効化／無効化します。
 *              RegisterGattOperationNotification()  を使用して登録していない GATT Service に含まれる GATT Characteristic に対して操作してはいけません。
 *              本 API によって有効化されるまで、デフォルトは無効化状態です。
 *              対象の GATT Characteristic が Client Characteristic Configuration Descriptor を持たず、GATT Server が任意に Notification/Indication を送信する場合でも、
 *              本 API によって有効化されるまでは、受信したデータは破棄されます。
 *              また、対象の GATT Characteristic が Client Characteristic Configuration Descriptor を持っており、
 *              かつ、GATT Server がその値に応じて、Notification/Indication の振る舞いを変える場合、WriteGattDescriptor() を使って値を書き込む必要があります。
 *
 * @param[in]   characteristic              Notification または Indication を有効化／無効化する Gatt Characteristic のオブジェクト
 * @param[in]   enable                      Notifiaction または Indication を有効にするかを示すフラグ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            操作に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultUnsupportedGattProperty, サポートされていないGATT Attribute Property です。}
 * @endretresult
 */
nn::Result EnableGattCharacteristicNotification(const GattCharacteristic& characteristic, bool enable) NN_NOEXCEPT;

/**
 * @brief       GATT Descriptor をRead します
 *
 * @details     RegisterGattOperationNotification()  を使用して登録していない GATT Service に含まれる GATT Descriptor に対して操作してはいけません。
 *              API の呼び出しが成功すると、実際に通信が行われます。
 *              通信の完了後、AcquireBleGattOperationEvent() で取得したイベントがシグナルされます。
 *              操作結果の operation には、@ref GattOperationType_ReadDescriptor が設定されています。
 *              受信したデータは、操作結果に含まれています。GattDescriptor::GetValue() を使用して確認することはできません。
 *              イベントがシグナルされるまで、同一の接続に対して GATT Attribute を操作することはできません。
 *
 * @param[in]   descriptor                  Read する Gatt Descriptor のオブジェクト
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            操作に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultGattBusy,                ビジー状態です。直前の操作の完了を待って再試行してください。}
 * @endretresult
 */
nn::Result ReadGattDescriptor(const GattDescriptor& descriptor) NN_NOEXCEPT;

/**
 * @brief       GATT Descriptor にWrite します
 *
 * @details     RegisterGattOperationNotification()  を使用して登録していない GATT Service に含まれるGATT Descriptor に対して操作してはいけません。
 *              API の呼び出しが成功すると、実際に通信が行われます。
 *              通信の完了後、AcquireBleGattOperationEvent() で取得したイベントがシグナルされます。
 *              操作結果の operation には、@ref GattOperationType_WriteDescriptor が設定されています。
 *              送信したデータは、操作結果に含まれていません。
 *              イベントがシグナルされるまで、同一の接続に対して GATT Attribute を操作することはできません。
 *
 * @param[in]   descriptor                  Write するGatt Descriptor のオブジェクト
 *
 * @pre
 *              - GattDescriptor::SetValue() を使用して、Write する値が設定されている
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            操作に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultGattBusy,                ビジー状態です。直前の操作の完了を待って再試行してください。}
 * @endretresult
 */
nn::Result WriteGattDescriptor(const GattDescriptor& descriptor) NN_NOEXCEPT;

//! @}

}}  // namespace nn::bluetooth
