﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       GATT Characteristic クラスの宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/bluetooth/bluetooth_GattAttribute.h>

namespace nn { namespace bluetooth {

    class GattService;

    class GattDescriptor;

    /**
     * @brief       GATT Characteristic を表すクラスです
     *
     * @details     GATT Attribute クラスを継承します
     */
    class GattCharacteristic : public GattAttribute
    {
    public:
        /**
         * @brief           コンストラクタです
         *
         * @details         全パラメータが無効な値で初期化された GATT Characteristic オブジェクトを生成します
         */
        GattCharacteristic() NN_NOEXCEPT;

        /**
         * @brief           コンストラクタです
         *
         * @details         与えられたパラメータで初期化された GATT Characteristic オブジェクトを生成します
         *
         * @param[in]       uuid                GATT Attribute UUID
         * @param[in]       handle              GATT Attribute Handle
         * @param[in]       connectionHandle    GATT Server との接続ハンドル
         * @param[in]       instanceId          Instance ID
         * @param[in]       property            Property
         */
        GattCharacteristic(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle,
                           uint16_t instanceId, uint8_t property) NN_NOEXCEPT;

        /**
         * @brief           デストラクタです
         */
        virtual ~GattCharacteristic() NN_NOEXCEPT;

        /**
         * @brief           Instance ID を取得します
         */
        uint16_t GetInstanceId() const NN_NOEXCEPT;

        /**
         * @brief           持っている全ての Gatt Attribute Property のビット単位の論理和を取得します
         */
        uint8_t GetProperties() const NN_NOEXCEPT;

        /**
         * @brief           自身が所属する GATT Service を取得します
         *
         * @param[out]      pOutService         GATT Service オブジェクトへのポインタ
         *
         * @pre
         *                  - pOutService != nullptr
         */
        void GetService(GattService* pOutService) const NN_NOEXCEPT;

        /**
         * @brief           自身に所属する GATT Descriptor を取得します
         *
         * @details         最大で @ref GattAttributeCountMaxClient 個の GATT Descriptor を取得します。
         *
         * @param[out]      pOutDescriptors             GATT Descriptor の配列へのポインタ
         * @param[in]       count                       pOutDescriptors の数
         *
         * @pre
         *                  - pOutDescriptors != nullptr
         *
         * @return          取得した GATT Descriptor の数
         */
        int GetDescriptors(GattDescriptor* pOutDescriptors, int count) const NN_NOEXCEPT;

        /**
         * @brief           自身に所属する GATT Descriptor のうち、指定した UUID を持つものを取得します
         *
         * @param[out]      pOutDescriptor              GATT Descriptor オブジェクトへのポインタ
         * @param[in]       uuid                        取得したい GATT Descriptor のUUID
         *
         * @pre
         *                  - pOutDescriptor != nullptr
         *
         * @return          GATT Descriptor が見つかれば true、見つからなければ false
         */
        bool GetDescriptor(GattDescriptor* pOutDescriptor, const GattAttributeUuid& uuid) const NN_NOEXCEPT;

        /**
         * @brief           接続相手に送信する値を設定します
         *
         * @details         以下の API によって送信される値を設定します。
         *                  - WriteGattCharacteristic()
         *
         *                  設定された値は、内部でコピーされます。
         *                  すでに値が設定されている場合、上書きします。
         *
         * @param[in]       pValue              設定する値
         * @param[in]       size                設定する値のサイズ
         *
         * @pre
         *                  - pValue != nullptr
         *                  - size <= @ref GattAttributeValueSizeMax
         */
        void SetValue(const void* pValue, size_t size) NN_NOEXCEPT;

        /**
         * @brief           接続相手に送信する値を取得します
         *
         * @details         以下の API によって送信される値を取得します。
         *                  - WriteGattCharacteristic()
         *
         * @param[out]      pOutValue           取得する値を格納するための配列へのポインタ
         * @param[in]       size                pOutValue のサイズ
         *
         * @pre
         *                  - pOutValue != nullptr
         *
         * @return          取得した値のサイズ
         */
        size_t GetValue(void* pOutValue, size_t size) const NN_NOEXCEPT;

    private:
        uint16_t    m_InstanceId;   //!< 同一の GATT Attribute UUID を持つ GATT Characteristic が、同一の GATT Server 上に存在する場合に、それらを識別するために用いられるサブ識別子（Instance ID）です

        uint8_t     m_Property;     //!< 自身への GATT 操作方法を表すビットフラグセット（GattAttributeProperty）です

        size_t      m_ValueLength;     //!< m_Value に設定されている値のサイズです

        uint8_t     m_Value[GattAttributeValueSizeMax];    //!< 接続相手に送信する値です
    };
}}  // namespace nn::bluetooth
