﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/service/bcat_ArchiveDecoder.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_StaticAssert.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>

namespace nn { namespace bcat { namespace service {

/*!
    @brief      BCAT アーカイブのダウンローダーです。
*/
class ArchiveDownloader
{
public:
    /*!
        @brief      メタデータの書き込み時に消費するジャーナルのサイズです。
    */
    static const int64_t MetaDataJournalSize = 48 * 1024;

    /*!
        @brief      ジャーナルサイズを指定する際の最小値です。
    */
    static const int64_t JournalSizeMin = MetaDataJournalSize + 16 * 1024;

    /*!
        @brief      ダウンロードバッファのサイズです。
    */
    static const size_t DownloadBufferSize = 512;

    /*!
        @brief      ダウンロードの進捗コールバックです。
    */
    typedef bool (*ProgressCallback)(int64_t now, int64_t total, void* param);

public:
    /*!
        @brief      コンストラクタです。
    */
    ArchiveDownloader() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~ArchiveDownloader() NN_NOEXCEPT;

    /*!
        @brief      リセットします。

        @details
                    本関数は、メンバー変数の初期化を行います。
    */
    void Reset() NN_NOEXCEPT;

    /*!
        @brief      パスを設定します。

        @param[in]  dataPath    データのパス。
        @param[in]  metaPath    メタデータのパス。

        @pre
            - dataPath != nullptr
            - metaPath != nullptr
            - データとメタデータのマウント名が一致している。
    */
    void SetPath(const char* dataPath, const char* metaPath) NN_NOEXCEPT;

    /*!
        @brief      パスフレーズを設定します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  passphrase  パスフレーズ。

        @pre
            - passphrase != nullptr
    */
    void SetPassphrase(const nn::ApplicationId& appId, const char* passphrase) NN_NOEXCEPT;

    /*!
        @brief      If-None-Match 用の ETag を設定します。

        @param[in]  eTag    ETag。

        @pre
            - eTag != nullptr

        @details
                    If-None-Match 用の ETag は新規ダウンロード時のみ有効です。
    */
    void SetETagForIfNoneMatch(const char* eTag) NN_NOEXCEPT;

    /*!
        @brief      ジャーナルサイズを設定します。

        @param[in]  size    ジャーナルサイズ。

        @pre
            - size == 0 || size >= JournalSizeMin

        @details
                    ジャーナルサイズに 0 以外を指定した場合、指定したサイズ分データを書き込む度にストレージをコミットします。@n
                    ジャーナルサイズに 0 を指定した場合、コミット処理は行われません。
    */
    void SetJournalSize(int64_t size) NN_NOEXCEPT;

    /*!
        @brief      プログレスコールバックを設定します。

        @param[in]  callback    コールバック。
        @param[in]  param       コールバックパラメーター。
    */
    void SetProgressCallback(ProgressCallback callback, void* param) NN_NOEXCEPT;

    /*!
        @brief      ファイル書き込み用のワークバッファを設定します。

        @param[in]  buffer  ワークバッファ。
        @param[in]  size    ワークバッファのサイズ。

        @pre
            - buffer != nullptr
            - size > 0
            - size % DownloadBufferSize == 0
    */
    void SetFileWorkBuffer(void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      ダウンロードします。

        @param[in]  url             URL。
        @param[in]  pCancelEvent    キャンセルイベント。

        @return     処理結果。

        @pre
            - url != nullptr
    */
    nn::Result Download(const char* url, nn::os::Event* pCancelEvent = nullptr) NN_NOEXCEPT;

    /*!
        @brief      ETag を取得します。

        @return     ETag。
    */
    const ETag& GetETag() const NN_NOEXCEPT;

private:
    //
    struct FileAccessParam
    {
        char mountName[nn::fs::MountNameLengthMax + 1];
        int64_t journal;
        int64_t journalWritten;
        const char* dataPath;
        const char* metaPath;
        nn::fs::FileHandle handle;
        bool isFileOpened;
    };

    //
    struct RequestParam
    {
        const char* eTagForIfNoneMatch;
        int64_t rangeBytes;
    };

    //
    struct ResponseParam
    {
        int statusCode;
        int64_t downloaded;
        int64_t rangeBytes;
    };

    //
    struct DecodeContext
    {
        ArchiveDecoder* pDecoder;
        Bit8* fileBuffer;
        Bit8* downloadBuffer;
        size_t fileBufferSize;
        size_t downloadBufferSize;
        size_t sizeInFileBuffer;
        size_t sizeInDownloadBuffer;
    };

private:
    //
    nn::ApplicationId m_AppId;
    const char* m_Passphrase;
    //
    FileAccessParam m_FileAccessParam;
    //
    RequestParam m_RequestParam;
    ResponseParam m_ResponseParam;
    //
    DecodeContext m_DecodeContext;
    //
    ProgressCallback m_ProgressCallback;
    void* m_ProgressCallbackParam;
    //
    ArchiveMetadata m_Meta;
    bool m_IsMetaDownloaded;
    //
    Bit8 m_DownloadBuffer[DownloadBufferSize];
    //
    nn::os::Event* m_pCancelEvent;
    //
    nn::Result m_LastError;

private:
    //
    nn::Result Open() NN_NOEXCEPT;
    //
    nn::Result OpenData() NN_NOEXCEPT;
    //
    nn::Result LoadMeta() NN_NOEXCEPT;
    //
    nn::Result SaveMeta() NN_NOEXCEPT;
    //
    nn::Result VerifyData() NN_NOEXCEPT;
    //
    nn::Result Preprocess(curl_slist** outHeaders, CURL* curl, const char* url) NN_NOEXCEPT;
    //
    nn::Result Perform(CURL* curl) NN_NOEXCEPT;
    //
    nn::Result Postprocess() NN_NOEXCEPT;
    //
    nn::Result FlushJournalIfFull(size_t writeSize) NN_NOEXCEPT;

private:
    //
    nn::Result HeaderCallback(const char* label, const char* value) NN_NOEXCEPT;
    //
    nn::Result HeaderEndCallback() NN_NOEXCEPT;
    //
    nn::Result ResponseBodyCallback(const char* buffer, size_t size) NN_NOEXCEPT;
    //
    static CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT;
    //
    static size_t HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static size_t HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static int HttpProgressCallback(void* param, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT;
};

}}}
