﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/msgpack/bcat_JsonPath.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace msgpack {

/*!
    @brief      整数型です。

    @details
                任意の整数型が格納されます。
*/
struct AnyInteger
{
    union
    {
        int64_t s;  //!< 64 ビット符号付き整数型。
        uint64_t u; //!< 64 ビット符号なし整数型。
    };
    bool isSigned;
};

/*!
    @brief      MessagePack イベントハンドラです。

    @details
                各 MessagePack データのイベントハンドルに特化した処理を記述します。@n
                MessagePack パーサーの利用者は、本クラスの派生を定義し、処理を記述してください。
*/
class MsgpackEventHandler
{
public:
    /*!
        @brief      コンストラクタです。
    */
    MsgpackEventHandler() NN_NOEXCEPT {}

    /*!
        @brief      デストラクタです。
    */
    virtual ~MsgpackEventHandler() NN_NOEXCEPT {}

    /*!
        @brief      オブジェクトの開始イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnStartObject(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      オブジェクトの終了イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnEndObject(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      配列の開始イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnStartArray(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      配列の終了イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnEndArray(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      キーの出現イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnKey(const JsonPath& jsonPath) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      値（null）の出現イベントです。

        @param[in]  jsonPath    JSON パス。

        @return     処理を継続するかどうか。
    */
    virtual bool OnNull(const JsonPath& jsonPath, std::nullptr_t) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        return true;
    }

    /*!
        @brief      値（bool）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnBoolean(const JsonPath& jsonPath, bool value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        return true;
    }

    /*!
        @brief      値（integer）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnInteger(const JsonPath& jsonPath, const AnyInteger& value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        return true;
    }

    /*!
        @brief      値（double）の出現イベントです。

        @param[in]  jsonPath    JSON パス。
        @param[in]  value       値。

        @return     処理を継続するかどうか。
    */
    virtual bool OnDouble(const JsonPath& jsonPath, double value) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        return true;
    }

    /*!
        @brief      値（string）の出現イベントです。

        @param[in]  jsonPath        JSON パス。
        @param[in]  value           値。
        @param[in]  length          文字列長。
        @param[in]  isOverflowed    オーバーフローフラグ。

        @return     処理を継続するかどうか。
    */
    virtual bool OnString(const JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);
        return true;
    }

    /*!
        @brief      値（binary）の出現イベントです。

        @param[in]  jsonPath        JSON パス。
        @param[in]  value           バイナリデータ。
        @param[in]  length          バイナリデータサイズ。
        @param[in]  isOverflowed    オーバーフローフラグ。

        @return     処理を継続するかどうか。
    */
    virtual bool OnBinary(const JsonPath& jsonPath, const void* value, size_t length, bool isOverflowed) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);
        return true;
    }

    /*!
        @brief      値（extension）の出現イベントです。

        @param[in]  jsonPath        JSON パス。
        @param[in]  value           拡張データ種別。
        @param[in]  value           拡張データ。
        @param[in]  length          拡張データサイズ。
        @param[in]  isOverflowed    オーバーフローフラグ。

        @return     処理を継続するかどうか。
    */
    virtual bool OnExtension(const JsonPath& jsonPath, int8_t type, const void* value, size_t length, bool isOverflowed) NN_NOEXCEPT
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(type);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);
        return true;
    }
};

/*!
    @brief      ログ出力するイベントハンドラです。
*/
class MsgpackPrintEventHandler : public MsgpackEventHandler
{
public:
    virtual bool OnStartArray(const JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);

        NN_SDK_LOG("%s:\n    [\n", jsonPath.ToString());

        return true;
    }

    virtual bool OnEndArray(const JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);

        NN_SDK_LOG("%s:\n    ]\n", jsonPath.ToString());

        return true;
    }

    virtual bool OnNull(const JsonPath& jsonPath, std::nullptr_t) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);

        NN_SDK_LOG("%s:\n    value = null\n", jsonPath.ToString());

        return true;
    }

    virtual bool OnBoolean(const JsonPath& jsonPath, bool value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        NN_SDK_LOG("%s:\n    value = %s\n", jsonPath.ToString(), value ? "true" : "false");

        return true;
    }

    virtual bool OnInteger(const JsonPath& jsonPath, const AnyInteger& value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        if (value.isSigned)
        {
            NN_SDK_LOG("%s:\n    value = %lld\n", jsonPath.ToString(), value.s);
        }
        else
        {
            NN_SDK_LOG("%s:\n    value = %llu\n", jsonPath.ToString(), value.u);
        }

        return true;
    }

    virtual bool OnDouble(const JsonPath& jsonPath, double value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);

        NN_SDK_LOG("%s:\n    value = %f\n", jsonPath.ToString(), value);

        return true;
    }

    virtual bool OnString(const JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);

        NN_SDK_LOG("%s:\n    value = %s\n", jsonPath.ToString(), value);

        return true;
    }

    virtual bool OnBinary(const JsonPath& jsonPath, const void* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);

        NN_SDK_LOG("%s:\n    value = binary, size=%zu\n", jsonPath.ToString(), length);

        return true;
    }

    virtual bool OnExtension(const JsonPath& jsonPath, int8_t type, const void* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(jsonPath);
        NN_UNUSED(type);
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(isOverflowed);

        NN_SDK_LOG("%s:\n    value = extension(%d), size=%zu\n", jsonPath.ToString(), type, length);

        return true;
    }
};

}}}}}
