﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/core/bcat_FileSystemConfig.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_Context.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      ファイルシステムライブラリのラッパーモジュールです。
*/
class FileSystem
{
public:
    /*!
        @brief      ジャーナル情報です。
    */
    struct JournalInfo
    {
        size_t written; //!< 書き込んだバイト数。
        size_t size;    //!< サイズ。
    };

public:
    /*!
        @brief      テストモードを有効にします。

        @details
                    テストモードが有効な場合、以下の API はストレージへのアクセスを行いません。

                    - FileSystem::Mount
                    - FileSystem::Unmount
                    - FileSystem::Commit

                    テスト実行前に必要とするストレージをマウントしてください。
    */
    static void EnableTestMode() NN_NOEXCEPT;

    /*!
        @brief      データ壊れをハンドリングする FS エラーハンドラーです。

        @param[in]  result  処理結果。

        @return     ABORT 指示子。
    */
    static nn::fs::AbortSpecifier ResultHandlerForHandlingDataCorruption(nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      ストレージをマウントします。

        @param[in]  mountName           マウント名。
        @param[in]  createIfNotExists   ストレージが存在しない時に作成するかどうか。

        @return     処理結果。
    */
    static nn::Result Mount(const char* mountName, bool createIfNotExists = false) NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンマウントします。

        @param[in]  mountName   マウント名。
    */
    static void Unmount(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをコミットします。

        @param[in]  mountName   マウント名。

        @return     処理結果。

        @details
                    ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    static nn::Result Commit(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @param[out] outSize     空き容量。
        @param[in]  mountName   マウント名。

        @return     処理結果。

        @pre
            - outSize != nullptr
    */
    static nn::Result GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に作成します。

        @param[in]  path        パス。
        @param[in]  isFilePath  ファイルパスを指定したかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    ファイルパスを指定した場合、親ディレクトリの生成を行います。
    */
    static nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath = false) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に削除します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    static nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;

    /*!
        @brief      ファイルを新規作成します。

        @param[in]  path            パス。
        @param[in]  size            ファイルサイズ。
        @param[in]  ignoreIfExists  ファイルがすでに存在する時に成功を返すかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    親ディレクトリが存在しない場合、親ディレクトリも作成します。
    */
    static nn::Result CreateFile(const char* path, int64_t size, bool ignoreIfExists = true) NN_NOEXCEPT;

    /*!
        @brief      ファイルをリネームします。

        @param[in]  currentPath 移動元のパス。
        @param[in]  newPath     移動先のパス。

        @return     処理結果。

        @pre
            - currentPath != nullptr
            - newPath != nullptr

        @details
                    親ディレクトリが存在しない場合、親ディレクトリも作成します。@n
                    また、移動先がすでに存在する場合、上書きします。
    */
    static nn::Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT;

    /*!
        @brief      ファイルサイズを取得します。

        @param[out] outSize ファイルサイズ。
        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - path != nullptr
    */
    static nn::Result GetFileSize(int64_t* outSize, const char* path) NN_NOEXCEPT;

    /*!
        @brief      エントリーが存在するかどうかを確認します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    static bool Exists(const char* path) NN_NOEXCEPT;

    /*!
        @brief      バッファを使用したファイルコピーを行います。

        @param[out] outJournalInfo  コピー後のジャーナル情報。
        @param[in]  source          コピー元のパス。
        @param[in]  dest            コピー先のパス。
        @param[in]  journalInfo     ジャーナル情報。
        @param[in]  buffer          バッファ。
        @param[in]  bufferSize      バッファサイズ。
        @param[in]  pCancelEvent    ファイルコピーのキャンセルイベント。

        @return     処理結果。

        @pre
            - outJournalInfo != nullptr
            - source != nullptr
            - dest != nullptr
            - journalInfo.size % BlockSize == 0
            - journalInfo.size > BlockSize
            - journalInfo.size > journalInfo.written
            - buffer != nullptr
            - bufferSize % BlockSize == 0
            - bufferSize > BlockSize

        @details
                    本関数は、コピー元のファイルを読み込みながらコピー先のファイルに書き込みます。@n
                    異なるストレージ間でのファイルコピーを実現します。

                    コピー中にジャーナル領域を使い切った場合、転送中であっても一度コミットが行われます。@n
                    この場合、コピー先のデータは完全ではないことに注意する必要があります。

                    キャンセルイベントを Signal した場合、コピー処理の完了を待たずに終了します。@n
                    この場合、ResultCanceledByUser が返ります。

                    親ディレクトリが存在しない場合、親ディレクトリも作成します。
    */
    static nn::Result CopyFileWithBuffer(JournalInfo* outJournalInfo, const char* source, const char* dest,
        JournalInfo journalInfo, void* buffer, size_t bufferSize, nn::os::Event* pCancelEvent = nullptr) NN_NOEXCEPT;

    /*!
        @brief      ストレージ上でのファイルサイズを計算します。

        @param[in]  size    ファイルサイズ。

        @return     ストレージ上でのファイルサイズ。
    */
    static int64_t CalculateFileSizeOnStorage(int64_t size) NN_NOEXCEPT;
};

}}}}}

/*!
    @brief      データ壊れをハンドリングすることを局所的に宣言します。
*/
#define NN_DETAIL_BCAT_HANDLING_DATA_CORRUPTION_SCOPED_DECLARE() \
    nn::fs::FsContext context_(detail::service::core::FileSystem::ResultHandlerForHandlingDataCorruption);  \
    nn::fs::ScopedFsContextSetter scopedContext_(context_)

/*!
    @brief      システム情報ストレージのマウント名です。
*/
#define NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME "bcat-sys"

/*!
    @brief      ダウンロードストレージのマウント名です。
*/
#define NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME "bcat-dl"

/*!
    @brief      ストレージを局所的にマウントします。

    @details
                各モジュールのロックとストレージのロックが競合してデッドロックが発生しないよう注意してください。
*/
#define NN_DETAIL_BCAT_STORAGE_SCOPED_MOUNT(mountName, createIfNotExists) \
    NN_RESULT_TRY(detail::service::core::FileSystem::Mount(mountName, createIfNotExists))   \
        NN_RESULT_CATCH(nn::fs::ResultDataCorrupted)                                        \
        {                                                                                   \
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);                       \
        }                                                                                   \
    NN_RESULT_END_TRY;                                                                      \
                                                                                            \
    NN_UTIL_SCOPE_EXIT                                                                      \
    {                                                                                       \
        detail::service::core::FileSystem::Unmount(mountName);                              \
    }

/*!
    @brief      システム情報ストレージを局所的にマウントします。
*/
#define NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT() \
    NN_DETAIL_BCAT_STORAGE_SCOPED_MOUNT(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME, true)

/*!
    @brief      システム情報ストレージを局所的にマウントします。（読み込み専用）
*/
#define NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT_READ_ONLY() \
    NN_DETAIL_BCAT_STORAGE_SCOPED_MOUNT(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME, false)

/*!
    @brief      ダウンロードストレージを局所的にマウントします。
*/
#define NN_DETAIL_BCAT_DOWNLOAD_STORAGE_SCOPED_MOUNT() \
    NN_DETAIL_BCAT_STORAGE_SCOPED_MOUNT(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME, true)
