﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>
#include <nn/bcat/detail/service/core/bcat_DiffView.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      データ配信キャッシュのダウンローダーです。
*/
class DeliveryCacheDownloader
{
public:
    /*!
        @brief      一時オブジェクトのサイズです。
    */
    static const size_t ObjectBufferSize = 64 * 1024;

public:
    /*!
        @brief      コンストラクタです。
    */
    DeliveryCacheDownloader() NN_NOEXCEPT;

    /*!
        @brief      ダウンロード処理のキャンセルイベントを設定します。

        @param[in]  pEvent  キャンセルイベント。

        @pre
            - pEvent != nullptr
    */
    void SetDownloadCancelEvent(nn::os::Event* pEvent) NN_NOEXCEPT;

    /*!
        @brief      同期処理のキャンセルイベントを設定します。

        @param[in]  pEvent  キャンセルイベント。

        @pre
            - pEvent != nullptr
    */
    void SetSyncCancelEvent(nn::os::Event* pEvent) NN_NOEXCEPT;

    /*!
        @brief      ダウンロードと同期の対象となるディレクトリを設定します。

        @param[in]  dirName ディレクトリ名。

        @pre
            - dirName != nullptr

        @details
                    本関数を使用した場合、配信リスト取得後の処理を一部スキップし、対象となるディレクトリのみダウンロードと同期を行います。
    */
    void SetTargetDirectory(const char* dirName) NN_NOEXCEPT;

    /*!
        @brief      ダウンロード処理を行います。

        @param[in]  appId                           アプリケーション ID。
        @param[in]  appVersion                      アプリケーションバージョン。
        @param[in]  ignoreNetworkUseRestriction     ネットワークの利用制限を無視するかどうか。
        @param[in]  enableProviderSpecificDelivery  事業者別配信が有効かどうか。
        @param[in]  buffer                          ワークバッファ。
        @param[in]  bufferSize                      ワークバッファのサイズ。

        @return     処理結果。

        @pre
            - taskId != InvalidTaskId
            - appId != nn::ApplicationId::GetInvalidId()
            - buffer != nullptr
            - bufferSize > 0
    */
    nn::Result Download(TaskId taskId, nn::ApplicationId appId, uint32_t appVersion,
        bool ignoreNetworkUseRestriction, bool enableProviderSpecificDelivery, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      進捗を取得します。

        @return     進捗。
    */
    const DeliveryCacheProgressImpl& GetProgress() const NN_NOEXCEPT;

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    TaskId m_TaskId;
    //
    nn::ApplicationId m_AppId;
    uint32_t m_AppVersion;
    //
    nn::os::Event* m_pDownloadCancelEvent;
    nn::os::Event* m_pSyncCancelEvent;
    //
    const char* m_TargetDirectory;
    //
    nn::nifm::NetworkConnection m_Connection;
    //
    char m_Passphrase[PassphraseLengthMax + 1];
    //
    ListDirectory m_Dirs[DeliveryCacheDirectoryCountMax];
    int m_DirCount;
    //
    DirectoryMetaView m_DirectoryMetaView;
    DirectoryDiffView m_DirectoryDiffView;
    //
    int m_IncreasingDirIndexes[DeliveryCacheDirectoryCountMax];
    int m_IncreasingDirCount;
    int m_ReducingDirIndexes[DeliveryCacheDirectoryCountMax];
    int m_ReducingDirCount;
    //
    DirectoryMeta m_DownloadDirectoryMeta;
    bool m_IsDownloadDirectoryMetaExists;
    //
    int m_ResumeDownloadDirectoryIndex;
    int m_DownloadTargetDirectoryIndex;
    //
    DeliveryCacheProgressImpl m_Progress;
    //
    Bit8* m_WorkBuffer;
    size_t m_WorkBufferSize;
    //
    Bit64 m_ObjectBuffer[ObjectBufferSize / sizeof (Bit64)];

private:
    //
    nn::Result StepSetup() NN_NOEXCEPT;
    nn::Result StepConnect(bool ignoreNetworkUseRestriction) NN_NOEXCEPT;
    nn::Result StepProcessList(bool enableProviderSpecificDelivery) NN_NOEXCEPT;
    nn::Result StepRemoveDirectories() NN_NOEXCEPT;
    nn::Result StepResumeDownloadDirectory() NN_NOEXCEPT;
    nn::Result StepDownloadTargetDirectory() NN_NOEXCEPT;
    nn::Result StepDownloadDirectories() NN_NOEXCEPT;
    //
    nn::Result ResumeDownloadDirectory(const ListDirectory& dir) NN_NOEXCEPT;
    nn::Result DownloadDirectory(const ListDirectory& dir) NN_NOEXCEPT;
    nn::Result SyncDirectory(const ListDirectory& dir) NN_NOEXCEPT;
    //
    nn::Result CleanupDownloadDirectory() NN_NOEXCEPT;
    nn::Result CleanupDownloadDirectoryIfNoneMatch(const char* root, const char* expectDirs[], int expectDirCount) NN_NOEXCEPT;
    //
    nn::Result LoadDownloadDirectoryMeta(DirectoryMeta* outMeta) NN_NOEXCEPT;
    nn::Result CreateDownloadDirectoryMetaFile(const DirectoryMeta& meta) NN_NOEXCEPT;
    nn::Result DeleteDownloadDirectoryMetaFile() NN_NOEXCEPT;
    //
    nn::Result GetResumeDownloadDirectoryIndex(int* outIndex) NN_NOEXCEPT;
    //
    nn::Result HandleResult(nn::Result result) NN_NOEXCEPT;
    //
    int SearchDirectoryIndex(const char* dirName) NN_NOEXCEPT;
    //
    bool IsCanceled() NN_NOEXCEPT;

private:
    static void LastProgressCallback(int64_t downloaded, const DirectoryName& dirName, void* param) NN_NOEXCEPT;
    static void StartDownloadFileCallback(const DirectoryName& dirName, const FileName& fileName, int64_t size, void* param) NN_NOEXCEPT;
    static void DownloadFileCallback(int64_t downloaded, void* param) NN_NOEXCEPT;
};

}}}}}
