﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/bcat_TypesDebug.h>
#include <nn/bcat/bcat_Result.h>

namespace nn { namespace bcat {

/*!
    @name タスク管理
    @{
*/

/*!
    @brief      バックグラウンドで動作するデータ配信タスクを列挙します。

    @param[out] outCount    取得したタスク情報数。
    @param[out] outInfos    タスク情報リスト。
    @param[in]  count       タスク情報リストの要素数。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - outInfos != nullptr
        - count > 0

    @details
                タスクの最大数は @ref TaskCountMax になります。
*/
nn::Result EnumerateBackgroundDeliveryTask(int* outCount, TaskInfo* outInfos, int count) NN_NOEXCEPT;

/*!
    @brief      指定したアプリケーションのデータ配信タスクの動作をブロックします。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                指定したアプリケーションのタスクがコミット処理中であっても、強制的に中断します。@n
                コミット処理中だったディレクトリは不定な状態になりますが、適切にクリーンアップ処理が行われます。

                ブロックされている間、バックグラウンド同期は行われません。@n
                また、即時同期要求も受け付けなくなります。

                @ref UnblockDeliveryTask を呼び出すか、システム再起動でブロックは解除されます。
*/
nn::Result BlockDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @brief      データ配信タスクのブロックを解除します。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()
*/
nn::Result UnblockDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @}
    @name データ配信
    @{
*/

/*!
    @brief      データ配信リストを取得します。

    @param[out] outSize データ配信リストのサイズ。
    @param[in]  buffer  データ配信リストを格納するバッファ。
    @param[in]  size    データ配信リストを格納するバッファのサイズ。
    @param[in]  appId   アプリケーション ID。

    @return     処理結果。
    @retval     ResultNotFound          データ配信リストが存在していない。
    @retval     ResultStorageNotFound   データ配信キャッシュストレージが存在していない。
    @retval     ResultLocked            データ配信キャッシュストレージはロックされている。

    @return     処理結果。

    @pre
        - outSize != nullptr
        - buffer != nullptr
        - size > 0
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                本関数はデバッグ専用 API です。@n
                製品アプリケーションで利用することはできません。
*/
nn::Result GetDeliveryList(size_t* outSize, void* buffer, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @brief      データ配信キャッシュストレージをクリアします。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。
    @retval     ResultStorageNotFound   データ配信キャッシュストレージが存在していない。
    @retval     ResultLocked            データ配信キャッシュストレージはロックされている。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                本関数はデバッグ専用 API です。@n
                製品アプリケーションで利用することはできません。
*/
nn::Result ClearDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @}
    @name ログ
    @{
*/

/*!
    @brief      最近受信したプッシュ通知のログを取得します。

    @param[out] outCount    取得した受信ログ数。
    @param[out] outLogs     受信ログリスト。
    @param[in]  count       受信ログリストの要素数。

    @return     処理結果。

    @pre
        - outCount != nullptr
        - outLogs != nullptr
        - count > 0

    @details
                システムを起動してから本関数の呼び出し時点までに受信したプッシュ通知のログを取得します。@n
                電源断によって受信ログは消去されます。

                受信ログは最近受信した順にソートされます。@n
                受信ログの数が最大数に到達している場合、古いものから削除されます。

                受信ログの最大数は @ref PushNotificationLogCountMax になります。
*/
nn::Result GetPushNotificationLog(int* outCount, PushNotificationLog* outLogs, int count) NN_NOEXCEPT;

/*!
    @}
*/

}}
