﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       オーディオコントローラのプレイレポートに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace audioctrl {

/**
 * @brief      PlayReport で送信するデバイスごとの音量情報
 */
struct PlayReportAudioVolume
{
    int8_t volume;
    bool mute;
};

/**
 * @brief      PlayReport で送信する音量情報
 */
struct PlayReportAudioVolumeData
{
    PlayReportAudioVolume speaker;
    PlayReportAudioVolume headphone;
    PlayReportAudioVolume usbOutputDevice;
    bool isHeadphonePowerLimited;
};

/**
 * @brief      PlayReport で送信する音声出力先情報
 */
enum PlayReportAudioOutputTarget
{
    PlayReportAudioOutputTarget_Invalid = 0,
    PlayReportAudioOutputTarget_Speaker,
    PlayReportAudioOutputTarget_Headphone,
    PlayReportAudioOutputTarget_Tv,
    PlayReportAudioOutputTarget_UsbOutputDevice,
    PlayReportAudioOutputTarget_UnsupportedUsbOutputDevice
};

/**
 * @brief
 *
 * @details     本体音量の更新を示す通知を指定のイベントオブジェクトにバインドします。
 *              本通知は本体音量が保存されるたびに通知が行われます。
 *              pEvent は nn::os::EventClearMode_AutoClear で初期化されます。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindAudioVolumeUpdateEventForPlayReport(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       PlayReport 用音量情報を返します
 *
 * @details     PlayReport 用音量情報を返します。
 *              本 API で取得できる値が更新された通知を受け取るためには BindAudioVolumeUpdateEventForPlayReport() を利用してください。
 *
 * @param[out]  pOutData
 *
 * @return      なし
 *
 * @pre
 *              - pOutValues != nullptr
 */
void GetAudioVolumeDataForPlayReport(PlayReportAudioVolumeData* pOutData) NN_NOEXCEPT;

/**
 * @brief
 *
 * @details     音声出力先情報の更新を示す通知を指定のイベントオブジェクトにバインドします。
 *              音声出力先情報の更新が発生するたびに通知が行われます。
 *              pEvent は nn::os::EventClearMode_AutoClear で初期化されます。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindAudioOutputDeviceUpdateEventForPlayReport(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       PlayReport 用音声出力先情報を返します
 *
 * @details     PlayReport 用音声出力先情報を返します。
 *              本 API で取得できる値が更新された通知を受け取るためには BindAudioOutputDeviceUpdateEventForPlayReport() を利用してください。
 *
 * @param[out]  pOutData
 *
 * @return      なし
 */
PlayReportAudioOutputTarget GetAudioOutputTargetForPlayReport() NN_NOEXCEPT;

}} // namespace nn::audioctrl
