﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       オーディオの出力先コントローラに関する公開ヘッダファイルです。
 */

#pragma once

#include <nn/audioctrl/audioctrl_AudioControllerTypes.h>
#include <nn/nn_TimeSpan.h>

namespace nn { namespace audioctrl {

/**
 * @internal
 * @brief           標準の出力先を指定します。
 * @param[in]       target
 * @param[in]       fadeOutTime
 * @param[in]       fadeInTime
 * @details
 * target で指定する出力先に標準の出力先を変更します。
 * 本関数の呼び出し完了後、以下の処理順序で出力先が切り替わります。
 *
 * 1. fadeOutTime で指定されたフェード時間をかけて、音声出力を無音化
 * 2. 標準出力先を切り替え
 * 3. fadeInTime で指定されたフェード時間をかけて、音声出力は元のボリュームまで復帰
 *
 * 上記処理の実行中に、この関数を再度呼び出した場合、
 * 実行中のフェード処理が完了した後に、再度 1. から処理を開始します。
 *
 * @note
 * - target に指定する値は AudioTarget_Tv もしくは AudioTarget_Speaker のみ許容されています。
 * - fadeOutTime, fadeInTime には任意の時間を指定可能ですが、ノイズの発生を抑えるために、それぞれの時間は所定の下限値にクランプされます。
 *   現在は 10ms に設定されています。
 */
void SetDefaultTarget(AudioTarget target, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT;

/**
 * @internal
 * @brief           標準の出力先を取得します
 * @return          出力先に設定されている AudioTarget を返します。
 * @details
 * 現在の標準出力先を取得します。
 *
 * SetDefaultTarget() によるフェード処理中に、この関数を呼んだ場合、
 * フェード処理中であっても最後に設定した標準出力先が返されます。
 */
AudioTarget GetDefaultTarget() NN_NOEXCEPT;

}}  // namespace nn::audioctrl
