﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Event.h>

namespace nn {
namespace audio {

/**
 * @internal
 * @brief       オーディオレンダラの高速レンダリングの有効無効を切り替えます。
 * @param[in]   enabled   有効無効設定、true で有効
 * @param[in]   pEvent    オーディオレンダラがレンダリングの完了を通知するための Event
 * @post
 * - enabled == GetFastRenderingMode()
 * @details
 * enabled に true を指定すると、実時間でのレンダリングから高速なレンダリングを行うモードが有効になります。
 * 有効状態では、オーディオレンダラは nn::audio::TriggerRendering() によってトリガされるまでレンダリングを待機します。
 * nn::audio::TriggerRendering() によってトリガされると、レンダリングが開始し pEvent に指定された nn::os::Event をシグナル状態にセットして、
 * nn::audio::TriggerRendering() によるトリガを待機します。
 *
 * 有効状態の間、オーディオレンダラはオーディオデバイスへの音声出力を停止します。
 * pEvent の登録は任意です。登録しない場合、オーディオレンダラは pEvent への操作をスキップします。
 * 無効状態にする場合は pEvent には nullptr を指定してください。
 *
 * @note
 * この関数は PC 環境でのみサポートされています。
 * 複数のオーディオレンダラ利用時の挙動は保証されません。
 */
void SetFastRenderingMode(bool enabled, nn::os::Event* pEvent) NN_NOEXCEPT;

/**
 * @internal
 * @brief       オーディオレンダラの高速レンダリングの有効無効状態を取得します。
 * @details
 * SetFastRenderingMode() で設定した有効無効状態を取得します。
 * 初期値は false です。
 *
 * @note
 * この関数は PC 環境でのみサポートされています。
 * 複数のオーディオレンダラ利用時の挙動は保証されません。
 */
bool GetFastRenderingMode() NN_NOEXCEPT;

/**
 * @internal
 * @brief       オーディオレンダラのレンダリング処理をトリガーします。
 * @details
 * この関数を呼ぶことで、オーディオレンダラのレンダリングを 1 オーディオフレーム実行します。
 * この関数の呼び出し後、1 フレーム分のレンダリングが完了すると nn::audio::SetFastRenderingMode() で pEvent として登録したイベントがシグナル状態になります。
 *
 * @note
 * この関数は PC 環境でのみサポートされています。
 * 複数のオーディオレンダラ利用時の挙動は保証されません。
 */
void TriggerRendering() NN_NOEXCEPT;

}  // namespace audio
}  // namespace nn
