﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/arp/detail/arp_IService.sfdl.h>
#include <nn/arp/detail/arp_RegistrarImpl.h>
#include <nn/arp/detail/arp_PropertyTable.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_IServiceObject.h>

namespace nn {
namespace arp {
namespace detail {

template <typename AllocatorType, size_t PropertyTableSize>
class ServiceImpl
{
private:
    typedef sf::ObjectFactory<typename AllocatorType::Policy> Factory;
    typedef PropertyTable<PropertyTableSize> PropertyTableType;

    AllocatorType& m_Allocator;
    PropertyTableType& m_PropertyTable;

public:
    ServiceImpl(AllocatorType& allocator, PropertyTableType& propertyTable) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_PropertyTable(propertyTable)
    {
    }
    Result AcquireRegistrar(sf::Out<nn::sf::SharedPointer<nn::arp::detail::IRegistrar>> pOut) NN_NOEXCEPT
    {
        // TORIAEZU Registrar が作られる時点で全レコードを削除する
        m_PropertyTable.Clear();

        auto p = Factory::template CreateSharedEmplaced<IRegistrar, RegistrarImpl<PropertyTableType>>(&m_Allocator, m_PropertyTable);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize());
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result DeleteProperties(const os::ProcessId& pid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pid != os::ProcessId::GetInvalidId(), ResultInvalidProcessId());
        m_PropertyTable.Delete(pid);
        NN_RESULT_SUCCESS;
    }
    Result GetApplicationLaunchProperty(sf::Out<nn::arp::ApplicationLaunchProperty> pOut, const os::ProcessId& pid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());
        NN_RESULT_THROW_UNLESS(pid != os::ProcessId::GetInvalidId(), ResultInvalidProcessId());
        NN_RESULT_THROW_UNLESS(
            m_PropertyTable.GetApplicationLaunchProperty(pOut.GetPointer(), pid),
            ResultNotRegistered());
        NN_RESULT_SUCCESS;
    }
    Result GetApplicationLaunchPropertyWithApplicationId(sf::Out<nn::arp::ApplicationLaunchProperty> pOut, const ApplicationId& applicationId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());
        NN_RESULT_THROW_UNLESS(applicationId != ApplicationId::GetInvalidId(), ResultNotRegistered());
        NN_RESULT_THROW_UNLESS(
            m_PropertyTable.GetApplicationLaunchProperty(pOut.GetPointer(), applicationId),
            ResultNotRegistered());
        NN_RESULT_SUCCESS;
    }
    Result GetApplicationControlProperty(sf::Out<nn::ns::ApplicationControlProperty> pOut, const os::ProcessId& pid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());
        NN_RESULT_THROW_UNLESS(pid != os::ProcessId::GetInvalidId(), ResultInvalidProcessId());
        NN_RESULT_THROW_UNLESS(
            m_PropertyTable.GetApplicationControlProperty(pOut.GetPointer(), pid),
            ResultNotRegistered());
        NN_RESULT_SUCCESS;
    }
    Result GetApplicationControlPropertyWithApplicationId(sf::Out<nn::ns::ApplicationControlProperty> pOut, const ApplicationId& applicationId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());
        NN_RESULT_THROW_UNLESS(applicationId != ApplicationId::GetInvalidId(), ResultNotRegistered());
        NN_RESULT_THROW_UNLESS(
            m_PropertyTable.GetApplicationControlProperty(pOut.GetPointer(), applicationId),
            ResultNotRegistered());
        NN_RESULT_SUCCESS;
    }
};

} // ~namespace nn::arp::detail
}
}
